#ifndef RLIB_DATAGRAM_MSGBUF
#define RLIB_DATAGRAM_MSGBUF 1

#include <cstdint>
#include <cstdlib>
#include <cassert>
#include <atomic>
#include <memory>
#include <stdexcept>
#include <rlib/stdio.hpp>

using std::size_t;

namespace rlib {
    namespace impl {
        template <typename ElementType>
        class thread_safe_queue {
            struct node {
                std::atomic<node *> next;
                ElementType *payload;
                explicit node(ElementType *payload, bool nonnull = true) : payload(payload), next(nullptr) {
                    if(nonnull)
                        assert(payload != nullptr);
                }
            };
            using apnode_t = std::atomic<node *>;

            // push to back (tail), pop from front (head).
            // To make it simple, the head node always exists. So pop will not hurt tail, and push will not change head.
            apnode_t head, tail;


            // List structure is the primary concern, we must keep the list structure consistent before taking care of head/tail ptr.
            void do_push_back(node *new_ele) {
                node *NULLPTR = nullptr;
                auto tail_node_ptr = tail.load();
                while(!std::atomic_compare_exchange_weak(&tail_node_ptr->next, &NULLPTR, new_ele)) {
                    // If failed to link the node to the tail, we catch up and try again.
                    NULLPTR = nullptr;
                    tail_node_ptr = tail.load();
                }

                // Move the tail ptr. If we succeed, that's good. (I don't think we might fail here)
                // TODO: prove that we will not fail here. Note: push-push scenario, we are safe.
                auto res = std::atomic_compare_exchange_strong(&tail, &tail_node_ptr, new_ele);
                assert(res);
            }
            ElementType *do_pop_front() {
                node *head_node_ptr;
                do {
                    head_node_ptr = head.load();
                    if(head_node_ptr->next == nullptr)
                        return nullptr;
                    // We move the head pointer forward to pop. (It will never invalidate tail node)
                    // If failed, just try again.
                } while(!std::atomic_compare_exchange_weak(&head, &head_node_ptr, head_node_ptr->next));

                // I love this idea: value stored in head->next->data, instead of head->data.
                auto payload_to_return = head_node_ptr->next.load()->payload;
                head_node_ptr->next.load()->payload = nullptr;

                assert(payload_to_return != nullptr);
                assert(*payload_to_return == 1);
                return payload_to_return;

                // TODO: delete head_node_ptr
            }
        public:
            thread_safe_queue()
                    : head(new node(nullptr, false)), tail(head.load()) {}
            void push(const ElementType &payload) {
                do_push_back(new node(new ElementType(payload)));
            }
            void push(ElementType &&payload) {
                do_push_back(new node(new ElementType(std::forward<ElementType>(payload))));
            }
            auto pop() {
                auto ret = do_pop_front();
                if(ret)
                    return std::shared_ptr<ElementType>(ret);
                else
                    throw std::out_of_range("pop empty queue");
            }
        };
    }
}

#endif

