using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text.RegularExpressions;

namespace FleetAGC.Activities.Util
{
    public static class BasicExprParser
    {
        public enum ItemType
        {
            DIGITS, LITERAL_STRING, IDENTIFIER, OP_ADD, OP_MIN, OP_MUL, OP_DIV, OP_MOD, PARENTHESIS_L, PARENTHESIS_R, SPACE, expr, hexpr, arglist, COMMA
        }
        public enum RuntimeType {INT, STRING, NONE, LIST}
        
        // C# only provides a noob strange non-standard regex. Let's prey that it would not break anything. 
        // C# don't allow you to run DFA-ready high-performance basic regex. fuck! 
        // Warning: Items should only match strings with non-zero length! 
        private static Dictionary<ItemType, string> Items = new()
        {
            {ItemType.DIGITS, "[0-9][0-9]*"},
            // "[^"\\]*(\\.[^"\\]*)*"  , which allowing escaped quotes in string. 
            {ItemType.LITERAL_STRING, "\"[^\"\\\\]*(\\\\.[^\"\\\\])*\""},
            {ItemType.IDENTIFIER, "[a-zA-Z_][0-9a-zA-Z_]*"},
            {ItemType.OP_ADD, "\\+"},
            {ItemType.OP_MIN, "-"},
            {ItemType.OP_MUL, "\\*"},
            {ItemType.OP_DIV, "/"},
            {ItemType.OP_MOD, "%"},
            {ItemType.PARENTHESIS_L, "\\("},
            {ItemType.PARENTHESIS_R, "\\)"},
            {ItemType.COMMA, ","},
            {ItemType.SPACE, "[ \\t\\n][ \\t\\n]*"}, // simply discarded in lexer
        };
        private static ItemType lexDiscarded = ItemType.SPACE;
        
        // This is a hand-written naive LL1 parser, so be careful while creating rules. 
        // You may have to add something to `evaluate` function, after adding new rules. 
        // That's why I love flex, but can not use it in C#. It's so simple&easy to use. 
        // 
        // Todo: Warning: I don't know how to set left-combination. DO NOT use minus/divide without parenthesis, I have warned you! 
        private static List<KeyValuePair<ItemType, List<ItemType>>> syntaxes = new()
        {
            new(ItemType.hexpr, new() {ItemType.IDENTIFIER, ItemType.PARENTHESIS_L, ItemType.arglist, ItemType.PARENTHESIS_R}),
            new(ItemType.hexpr, new() {ItemType.PARENTHESIS_L, ItemType.expr, ItemType.PARENTHESIS_R}),
            new(ItemType.hexpr, new() {ItemType.DIGITS}),
            new(ItemType.hexpr, new() {ItemType.LITERAL_STRING}),
            new(ItemType.expr, new() {ItemType.expr, ItemType.OP_ADD, ItemType.expr}),
            new(ItemType.expr, new() {ItemType.expr, ItemType.OP_MIN, ItemType.expr}),
            new(ItemType.hexpr, new() {ItemType.hexpr, ItemType.OP_MUL, ItemType.hexpr}),
            new(ItemType.hexpr, new() {ItemType.hexpr, ItemType.OP_DIV, ItemType.hexpr}),
            new(ItemType.hexpr, new() {ItemType.hexpr, ItemType.OP_MOD, ItemType.hexpr}),
            new(ItemType.expr, new() {ItemType.hexpr}),
            new(ItemType.arglist, new() {ItemType.expr}),
            new(ItemType.arglist, new() {ItemType.arglist, ItemType.COMMA, ItemType.expr}),
        };
        private static ItemType syntax_root = ItemType.expr;
        
        ///////////////////////////////// Syntax Definition End //////////////////////////////////////////////////

        private static void TypeCheck(bool what, string msg = "")
        {
            if (!what)
                throw new ArgumentException("TypingException: " + msg);
        }

        public class LexingResultType
        {
            public ItemType type;
            public string text;
        }

        private static List<LexingResultType> lex(string input)
        {
            var compiled_tokens = Items.ToDictionary(kv => kv.Key, kv => new Regex("\\G"+kv.Value, RegexOptions.Compiled));
            var curr_position = 0;
            var results = new List<LexingResultType>();
            while (curr_position < input.Length)
            {
                var matches = compiled_tokens
                    .ToDictionary(kv => kv.Key, kv => kv.Value.Match(input, curr_position))
                    .Where(kv => kv.Value.Success);

                var best_match = matches.First(); // throws on lexing error: invalid syntax. 
                foreach (var match in matches)
                {
                    if (match.Value.Length > best_match.Value.Length)
                        best_match = match;
                }

                curr_position += best_match.Value.Length;
                if(best_match.Key != lexDiscarded)
                    results.Add(new LexingResultType {type = best_match.Key, text = best_match.Value.Value});
            }

            return results;
        }

        public class EvaluationResultType
        {
            // Operators, functions currently have no runtime type. 
            public RuntimeType type;
            // C# doesn't support meta-programming at all. So I have to use this. 
            public int valueInt;
            public string valueString;
            public List<EvaluationResultType> valueList;
            public string originalText; // for debugging and error message
        }

        /*
         * This function is tied to syntax rules. Don't forget to modify this function while updating rules. 
         */
        private static EvaluationResultType evaluateOneLexingResult(LexingResultType lexingResult)
        {
            var result = new EvaluationResultType();
            if (lexingResult.type == ItemType.DIGITS)
            {
                result.type = RuntimeType.INT;
                result.valueInt = int.Parse(lexingResult.text);
            }
            else if (lexingResult.type == ItemType.LITERAL_STRING)
            {
                result.type = RuntimeType.STRING;
                result.valueString = lexingResult.text.Substring(1, lexingResult.text.Length - 2);
                result.valueString = result.valueString.Replace("\\\"", "\""); // default string delimiter works. 
            }
            else
            {
                result.type = RuntimeType.NONE;
            }
            result.originalText = lexingResult.text;
            return result;
        }
        /*
         * This function is tied to syntax rules. Don't forget to modify this function while updating rules. 
         */
        private static EvaluationResultType evaluate(KeyValuePair<ItemType, List<ItemType>> rule_to_use, List<EvaluationResultType> args)
        {
            var result = new EvaluationResultType();
            // I can only use this naive switch-case in some of LL1 syntax. 
            switch (rule_to_use.Value[0])
            {
                case ItemType.expr:
                case ItemType.hexpr:
                    if (rule_to_use.Value.Count == 1)
                    {  // expr ::= hexpr, arglist ::= expr
                        if (rule_to_use.Key == ItemType.arglist)
                        {
                            result.type = RuntimeType.LIST;
                            result.valueList = new() {args[0]};
                        }
                        else
                            result = args[0];
                        break;
                    }
                    switch (rule_to_use.Value[1])
                    {
                        case ItemType.OP_ADD:
                            TypeCheck(args[0].type == args[2].type,
                                $"TypeCheck error: `{args[0].originalText}` and `{args[2].originalText}` must have the same type. ");
                            result = args[0];
                            if (result.type == RuntimeType.INT)
                                result.valueInt += args[2].valueInt;
                            else
                                result.valueString += args[2].valueString;
                            break;
                        case ItemType.OP_MIN:
                            TypeCheck(args[0].type == args[2].type,
                                $"TypeCheck error: `{args[0].originalText}` and `{args[2].originalText}` must have the same type. ");
                            TypeCheck(args[0].type == RuntimeType.INT, $"TypeCheck error: `{args[0].originalText}` must have INT type. ");
                            result = args[0];
                            result.valueInt -= args[2].valueInt;
                            break;
                        case ItemType.OP_MUL:
                            TypeCheck(args[0].type == args[2].type,
                                $"TypeCheck error: `{args[0].originalText}` and `{args[2].originalText}` must have the same type. ");
                            TypeCheck(args[0].type == RuntimeType.INT, $"TypeCheck error: `{args[0].originalText}` must have INT type. ");
                            result = args[0];
                            result.valueInt *= args[2].valueInt;
                            break;
                        case ItemType.OP_DIV:
                            TypeCheck(args[0].type == args[2].type,
                                $"TypeCheck error: `{args[0].originalText}` and `{args[2].originalText}` must have the same type. ");
                            TypeCheck(args[0].type == RuntimeType.INT, $"TypeCheck error: `{args[0].originalText}` must have INT type. ");
                            result = args[0];
                            result.valueInt /= args[2].valueInt;
                            break;
                        case ItemType.OP_MOD:
                            TypeCheck(args[0].type == args[2].type,
                                $"TypeCheck error: `{args[0].originalText}` and `{args[2].originalText}` must have the same type. ");
                            TypeCheck(args[0].type == RuntimeType.INT, $"TypeCheck error: `{args[0].originalText}` must have INT type. ");
                            result = args[0];
                            result.valueInt %= args[2].valueInt;
                            break;
                    }
                    break;
                case ItemType.DIGITS:
                case ItemType.LITERAL_STRING:
                    result = args[0];
                    break;
                case ItemType.PARENTHESIS_L:
                    // sexpr ::= ( expr )
                    result = args[1];
                    break;
                case ItemType.IDENTIFIER:
                    TypeCheck(args[2].type == RuntimeType.LIST);
                    result = BasicExprBuiltins.evaluate(args[0].originalText, args[2].valueList);
                    break;
                case ItemType.arglist:
                    // arglist ::= arglist , expr
                    TypeCheck(args[0].type == RuntimeType.LIST);
                    result = args[0];
                    result.valueList.Add(args[2]);
                    break;
                default:
                    throw new InvalidProgramException("Didn't evaluate rule " + rule_to_use.ToString());
            }
            result.originalText = args.Aggregate("", (s, ele) => s + ele.originalText);
            return result;
        }

        private static Dictionary<ItemType, List<ItemType>> heads_cache = new();
        /*
         * cached function: get the `heads` of a type, so that LL1 algorithm could
         *   determine, which rule should it select. 
         * The result should include `item` itself, because implicit rule `a ::= a`
         *   is considered. 
         */
        private static List<ItemType> get_heads(ItemType item)
        {
            if (heads_cache.ContainsKey(item)) return heads_cache[item];
            var results = new List<ItemType>();
            var results2 = new List<ItemType>() {item}; // include `item` itself here. 
            foreach (var head in syntaxes.Where(kv => kv.Key == item).Select(kv => kv.Value[0]))
            {
                if(!results.Contains(head) && head != item) results.Add(head);
            }
            foreach (var result in results)
            {
                foreach (var head in get_heads(result))
                {
                    // They don't want to be modified on-the-fly...
                    if(!results.Contains(head)) results2.Add(head);
                }
            }
            results = results.Concat(results2).ToList();
            heads_cache.Add(item, results);
            return results;
        }
        
        /*
         * This function only cares about parsing. Work about evaluation is in a dedicated function. 
         * returns: consumed tokens count (for parsing), and the evaluation result (for evaluation).
         *
         * This parser contains a patch to LL1 to allow left-recursive:
         * For example, if we have:
         * (1) e ::= e p q ...
         * (2) e ::= a ...
         * (3) e ::= b ...
         * And when we're expecting an e, we firstly evaluate e without (1), and then add a while loop to
         *   consume all `p q ...`. Once we notice a failure in the while loop, we stop and leave.
         *
         * That's why we have `patch_left_recursive`. 
         *   In the first matching, this function ignores all left-recursive rules like (1) to avoid a crash.
         *   Then, we add an additional while loop, to continue consuming `p q ...`,
         *     and collect all of them for another evaluation on rule (1).
         *
         * However, this solution also cause other issue:
         * For example, if we have:
         * (1) e ::= e p q
         * (2) res ::= e p q
         * (3) e ::= a
         * Now `a` should match `res`, but it failed. because the left-recursive rule has already ate `p q`. 
         */
        private static (int, EvaluationResultType) parseAndEvaluateImpl(List<LexingResultType> input, int begin_index, ItemType expected)
        {
            (int, EvaluationResultType) result;
            if (begin_index >= input.Count)
                throw new SyntaxErrorException("Expecting " + expected + ", got EOF. ");
            if (expected == input[begin_index].type)
            {
                // leaf node. Just yield and consume one token. 
                result = (1, evaluateOneLexingResult(input[begin_index]));
            }
            else
            {
                var matched_rules = syntaxes.Where(kv => kv.Key == expected && get_heads(kv.Value[0]).Contains(input[begin_index].type) && kv.Value[0] != kv.Key);
                if (matched_rules.Count() < 1)
                    throw new SyntaxErrorException("Syntax error. Expecting " + expected + ", unable to find a rule to deduce expected type. ");
                if (matched_rules.Count() > 1)
                    throw new InvalidProgramException("Syntax definition error. Expecting " + expected + ", but matched ambiguous rules. Check syntax definition in CSharp source, and make sure it's LL1. ");
                var rule_to_use = matched_rules.First();
                
                // Execute the selected rule
                var curr_index = begin_index;
                var subele_results = new List<EvaluationResultType>();
                foreach (var expected_subele in rule_to_use.Value)
                {
                    var (consumed, evaluationResult) = parseAndEvaluateImpl(input, curr_index, expected_subele);
                    curr_index += consumed;
                    subele_results.Add(evaluationResult);
                }
                result = (curr_index - begin_index, evaluate(rule_to_use, subele_results));
            }
            
            // Patch: try executing left-recursive rules. Now an `e` has already been matched, deal with left-recursive. 
            var lr_rules = syntaxes.Where(kv => kv.Key == expected && kv.Value[0] == kv.Key);
            while (true)
            {
                bool hit_good_rule = false;
                foreach (var rule in lr_rules)
                {
                    try
                    {
                        var curr_index = begin_index + result.Item1;
                        var subele_results = new List<EvaluationResultType>();
                        foreach (var element in rule.Value.Skip(1))
                        {
                            var (consumed, evaluationResult) = parseAndEvaluateImpl(input, curr_index, element);
                            curr_index += consumed;
                            subele_results.Add(evaluationResult);
                        }

                        // Wtf? We survived this? 
                        hit_good_rule = true;
                        result.Item1 = curr_index - begin_index;
                        subele_results.Insert(0, result.Item2); // push_front
                        result.Item2 = evaluate(rule, subele_results);
                        // Since we survived this, we should break the loop, and continue the next round of try-match. 
                        break;
                    }
                    catch (SyntaxErrorException) { }
                }
                if (!hit_good_rule)
                {
                    // We didn't hit any good rule. It's time to stop trying. 
                    break;
                }
            }
            return result;
        }

        public static EvaluationResultType execute(string script)
        {
            var lexingRes = lex(script);
            var (_, evaluationRes) = parseAndEvaluateImpl(lexingRes, 0, syntax_root);
            return evaluationRes;
        }
    }
}

