package main

import (
	"errors"
	"github.com/mcuadros/go-version"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
	"sync"
)

// Scan a directory recursively, and parse `**/*.csproj`, to list all referenced nuget packages. (in CoreXT reference format)
func ScanForDependencies(scanPath string) (allDeps []dependencyItem) {
	// Scan for dependencies
	files, err := ScanCsprojInDirectory(scanPath)
	logErrorIfAny(err, "ScanCsprojInDirectory")

	for _, file := range files {
		dependencies, err := ParseDependencies(file)
		logErrorIfAny(err, "ParseDependencies " + file)
		allDeps = append(allDeps, dependencies...)
	}
	return
}

// Download `allDeps` to localRepoPath, with nuget.config from nugetConfigPath.
// However, nugetConfigPath is not supported now. We use default nuget.config now. (Usually in ~/.nuget/)
func SyncPackages(nugetConfigPath, localRepoPath string, allDeps []dependencyItem) {
	// Prepare nuget local repo
	depsToSync := depsDeduplicate(allDeps)

	var wg sync.WaitGroup
	concurrencyLimitChan := make(chan int, 64)
	for index_, dep_ := range depsToSync {
		wg.Add(1)
		concurrencyLimitChan <- 1 // will block if there's already 64 goroutine running
		go func(index int, dep dependencyItem) {
			defer func() {<-concurrencyLimitChan ; wg.Done()}()
			log.Printf("[%v/%v] Begin downloading: %v:%v as var %v", index, len(depsToSync), dep.pkgName, dep.targetNetVer, dep.envName)

			realPkgName, realPkgVer := extractPostfixPkgVerFromPkgName(dep.pkgName)
			if realPkgVer == "" {
				// This is not a pkgname.pkgver format. Use targetNetVer to deduce.
				realPkgVer = dep.targetNetVer
			}

			log.Println("EXEC: nuget-download-package", realPkgName, realPkgVer, localRepoPath, nugetConfigPath)
			cmd := exec.Command("nuget-download-package", realPkgName, realPkgVer, localRepoPath, nugetConfigPath)
			stdout, err := cmd.CombinedOutput()
			log.Println(string(stdout))
			// We can do nothing on failure. usually, if envName is NULL, we need not install it at all.
			logErrorIfAny(err, "EXEC command")

			log.Printf("[%v/%v] End downloading: %v:%v as var %v", index, len(depsToSync), dep.pkgName, dep.targetNetVer, dep.envName)
		}(index_, dep_)
	}
	wg.Wait()
}

// Given package name, this function lookup the localRepoPath, and returns path to a good package version.
// The returned pkgPath is used to populate environment variables.
// |
// The pkgName is case in-sensitive.
func GetPackagePathFromName(localRepoPath, pkgName, targetNetVer string) (pkgPath string, err error) {
	guessBase := localRepoPath + string(os.PathSeparator) + strings.ToLower(pkgName)
	files, err := ioutil.ReadDir(guessBase + string(os.PathSeparator))
	if err != nil {
		return
	}

	maxVersion := ""
	for _, f := range files {
		if f.IsDir() && pathIsDir(guessBase + string(os.PathSeparator) + f.Name() + string(os.PathSeparator) + "lib" + string(os.PathSeparator) + targetNetVer) {
			if version.CompareSimple(f.Name(), maxVersion) >= 0 {
				maxVersion = f.Name()
			}
		}
	}

	if maxVersion != "" {
		return guessBase + string(os.PathSeparator) + maxVersion, nil
	} else {
		return "", errors.New("No version found for package " + pkgName + ":" + targetNetVer)
	}
}

