package main

import (
	"errors"
	"fmt"
	"github.com/mcuadros/go-version"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
)

// Scan a directory recursively, and parse `**/*.csproj`, to list all referenced nuget packages. (in CoreXT reference format)
func ScanForDependencies(scanPath string) (allDeps []dependencyItem) {
	// Scan for dependencies
	files, err := ScanCsprojInDirectory(scanPath)
	logErrorIfAny(err, "ScanCsprojInDirectory")

	for _, file := range files {
		dependencies, err := ParseDependencies(file)
		logErrorIfAny(err, "ParseDependencies " + file)
		allDeps = append(allDeps, dependencies...)
	}
	return
}

// Download `allDeps` to localRepoPath, with nuget.config from nugetConfigPath.
// However, nugetConfigPath is not supported now. We use default nuget.config now. (Usually in ~/.nuget/)
func SyncPackages(nugetConfigPath, localRepoPath string, allDeps []dependencyItem) {
	log.Println("WARNING: We do not support custom nugetConfigPath yet. Please make sure your default nuget.config works. (usually ~/.nuget/NuGet/NuGet.Config)")

	// We must create a fake environment to make dotnet happy. Let's do that!
	// Create the fake environment directory.
	fakeProjectTemplate := "<Project Sdk=\"Microsoft.NET.Sdk\"><PropertyGroup><TargetFramework>%v</TargetFramework></PropertyGroup></Project>"
	fakeProjectDir, err := ioutil.TempDir("", "openxt-syncpkg-tmp")
	panicErrorIfAny(err, "createTempDir for sync package")
	defer os.RemoveAll(fakeProjectDir)

	// Prepare nuget local repo
	for _, dep := range allDeps {
		log.Println("Downloading pkg: " + dep.pkgName + ":" + dep.targetNetVer + " as var " + dep.envName)
		// Create the fake project.
		err = ioutil.WriteFile(filepath.Join(fakeProjectDir, "noob.csproj"), []byte(fmt.Sprintf(fakeProjectTemplate, dep.targetNetVer)), 0666)
		logErrorIfAny(err, "write noob.csproj for " + dep.pkgName)

		// dotnet add package Microsoft.Azure.Cis.Activities.Contracts --package-directory ~/tmp/your.repo # --framework is not working at all.
		cmd := exec.Command("dotnet", "add", "package", dep.pkgName, "--package-directory", localRepoPath)
		cmd.Dir = fakeProjectDir
		stdout, err := cmd.Output()
		log.Println(string(stdout))
		// We can do nothing on failure. usually, if envName is NULL, we need not install it at all.
		logErrorIfAny(err, "EXEC command")
	}
}

// Given package name, this function lookup the localRepoPath, and returns path to a good package version.
// The returned pkgPath is used to populate environment variables.
// |
// The pkgName is case in-sensitive.
func GetPackagePathFromName(localRepoPath, pkgName, targetNetVer string) (pkgPath string, err error) {
	guessBase := localRepoPath + string(os.PathSeparator) + strings.ToLower(pkgName)
	files, err := ioutil.ReadDir(guessBase + string(os.PathSeparator))
	if err != nil {
		return
	}

	maxVersion := ""
	for _, f := range files {
		if f.IsDir() && pathIsDir(guessBase + string(os.PathSeparator) + f.Name() + string(os.PathSeparator) + "lib" + string(os.PathSeparator) + targetNetVer) {
			if version.CompareSimple(f.Name(), maxVersion) >= 0 {
				maxVersion = f.Name()
			}
		}
	}

	if maxVersion != "" {
		return guessBase + string(os.PathSeparator) + maxVersion, nil
	} else {
		return "", errors.New("No version found for package " + pkgName + ":" + targetNetVer)
	}
}

