package main

import (
	"errors"
	"github.com/mcuadros/go-version"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
	"sync"
)

// Scan a directory recursively, and parse `**/*.csproj`, to list all referenced nuget packages. (in CoreXT reference format)
func ScanForDependencies(scanPath string) (allDeps []dependencyItem) {
	// Firstly: Find all .csproj files, recursively.
	files, err := FindInDirectory(scanPath, "^.+\\.(csproj)$")
	logErrorIfAny(err, "ScanCsprojInDirectory")

	// Secondly: parse them for deps.
	for _, file := range files {
		dependencies, err := ParseDependencies(file)
		logErrorIfAny(err, "ParseDependencies " + file)
		allDeps = append(allDeps, dependencies...)
	}
	return
}
// Scan a directory recursively, and parse `**/packages.config`, to list all referenced nuget packages. (in nuget standard format)
func ScanForDependencies2(scanPath string) (allDeps []dependencyItemStripped) {
	// Firstly, scan for all packages.config, recursively.
	files, err := FindInDirectory(scanPath, "^packages.config$")
	logErrorIfAny(err, "Scan packages.config in directory")

	// Secondly, parse them for deps.
	for _, file := range files {
		log.Printf("Downloading packages from %v", file)
		deps, err := ParsePackagesConfig(file)
		logErrorIfAny(err, "Parse packages config")

		allDeps = append(allDeps, deps...)
	}
	return
}


// Download `allDeps` to localRepoPath, with nuget.config from nugetConfigPath.
// However, nugetConfigPath is not supported now. We use default nuget.config now. (Usually in ~/.nuget/)
// Update: nugetConfigPath is support after migrated to nuget-download-package.
func SyncPackages(nugetConfigPath, localRepoPath string, allDeps []dependencyItem) {
	allDeps = depsDeduplicate(allDeps)

	// Convert dependencyItem(for CoreXT) to dependencyItemStripped(for nuget-package-download)
	depsStripped := []dependencyItemStripped{}
	for _, dep := range allDeps {
		realPkgName, pkgVerOrNetVer := extractPostfixPkgVerFromPkgName(dep.pkgName)
		if pkgVerOrNetVer == "" {
			// This is not a pkgname.pkgver format. Use targetNetVer to deduce.
			pkgVerOrNetVer = dep.targetNetVer
		}
		depsStripped = append(depsStripped, dependencyItemStripped{realPkgName, pkgVerOrNetVer})
	}

	SyncPackages2(nugetConfigPath, localRepoPath, depsStripped)
}

// This function does the actual job, to download packages.
func SyncPackages2(nugetConfigPath, localRepoPath string, allDeps []dependencyItemStripped) {
	allDeps = depsDeduplicate2(allDeps)

	var wg sync.WaitGroup
	concurrencyLimitChan := make(chan int, 64)
	for index_, dep_ := range allDeps {
		wg.Add(1)
		concurrencyLimitChan <- 1 // will block if there's already 64 goroutine running
		go func(index int, dep dependencyItemStripped) {
			defer func() {<-concurrencyLimitChan ; wg.Done()}()
			log.Printf("[%v/%v] Begin downloading: %v:%v", index, len(allDeps), dep.pkgName, dep.pkgVerOrNetVer)
			log.Println("EXEC: nuget-download-package", dep.pkgName, dep.pkgVerOrNetVer, localRepoPath, nugetConfigPath)
			cmd := exec.Command("nuget-download-package", dep.pkgName, dep.pkgVerOrNetVer, localRepoPath, nugetConfigPath)
			stdout, err := cmd.CombinedOutput()
			log.Println(string(stdout))
			// We can do nothing on failure. usually, if envName is NULL, we need not install it at all.
			logErrorIfAny(err, "EXEC command")

			log.Printf("[%v/%v] End downloading: %v:%v", index, len(allDeps), dep.pkgName, dep.pkgVerOrNetVer)
		}(index_, dep_)
	}
	wg.Wait()
}



// Given package name, this function lookup the localRepoPath, and returns path to a good package version.
// The returned pkgPath is used to populate environment variables.
// |
// The pkgName is case in-sensitive.
func GetPackagePathFromName(localRepoPath, pkgName, targetNetVer string) (pkgPath string, err error) {
	realPkgName, pkgVerHint := extractPostfixPkgVerFromPkgName(pkgName) // pkgName with pkgVer should be discarded.
	guessBase := localRepoPath + string(os.PathSeparator) + strings.ToLower(realPkgName)
	files, err := ioutil.ReadDir(guessBase + string(os.PathSeparator))
	if err != nil {
		return
	}

	maxVersion := ""
	for _, f := range files {
		if f.IsDir() && pathIsDir(guessBase + string(os.PathSeparator) + f.Name() + string(os.PathSeparator) + "lib" + string(os.PathSeparator) + targetNetVer) {
			if strings.HasPrefix(maxVersion, pkgVerHint) && ! strings.HasPrefix(f.Name(), pkgVerHint) {
				// If he matches pkgVerHint but you don't, you have no chance to be a candidate.
				continue
			}
			if version.CompareSimple(f.Name(), maxVersion) >= 0 {
				maxVersion = f.Name()
			}
		}
	}

	if maxVersion != "" {
		return guessBase + string(os.PathSeparator) + maxVersion, nil
	} else {
		return "", errors.New("No version found for package " + pkgName + ":" + targetNetVer)
	}
}

