package main

import (
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"reflect"
	"regexp"
	"strings"
)

func Filter(arr interface{}, cond func(interface{}) bool) interface{} {
	contentType := reflect.TypeOf(arr)
	contentValue := reflect.ValueOf(arr)

	newContent := reflect.MakeSlice(contentType, 0, 0)
	for i := 0; i < contentValue.Len(); i++ {
		if content := contentValue.Index(i); cond(content.Interface()) {
			newContent = reflect.Append(newContent, content)
		}
	}
	return newContent.Interface()
}

func logErrorIfAny(err error, title string) {
	if err != nil {
		log.Println("[Error] on " + title + ": " + err.Error())
	}
}
func panicErrorIfAny(err error, title string) {
	if err != nil {
		panic("[Error] on " + title + ": " + err.Error())
	}
}

func stringToTempFile(s, pattern string) (filename string, err error) {
	tmpfile, err := ioutil.TempFile("", pattern)
	if err != nil {
		return
	}

	_, err = tmpfile.Write([]byte(s))
	if err != nil {
		return
	}
	err = tmpfile.Close()
	if err != nil {
		return
	}
	return tmpfile.Name(), nil
}

func assertStringNotEmpty(s, msg string) {
	if s == "" {
		panic(msg)
	}
}

// Example:
// PkgCis_SdkV2 => Cis.SdkV2
// PkgMicrosoft_Azure_Management_Cdn_Fluent => Microsoft.Azure.Management.Cdn.Fluent
// On error, returns empty string.
func guessPkgNameFromVarName(varName string) string {
	if len(varName) <= 3 || varName[:3] != "Pkg" {
		return ""
	}
	return strings.ReplaceAll(varName[3:], "_", ".")
}

func pathIsDir(path string) bool {
	_, err := os.Stat(path)
	if err == nil { return true }
	if os.IsNotExist(err) { return false }
	return false
}

// On error, return empty string
func targetFrameworkVersion2TargetFramework(oldVersion string) string {
	if oldVersion[:1] != "v" {
		return ""
	}
	return "net" + strings.ReplaceAll(oldVersion[1:], ".", "")
}

func depsDeduplicate(deps []dependencyItem) (resultDeps []dependencyItem) {
	hashset := make(map[string]bool)
	for _, dep := range deps {
		niddle := dep.envName + dep.targetNetVer
		if _, ok := hashset[niddle]; !ok {
			hashset[niddle] = true
			resultDeps = append(resultDeps, dep)
		}
	}
	return
}
func depsDeduplicate2(deps []dependencyItemStripped) (resultDeps []dependencyItemStripped) {
	hashset := make(map[string]bool)
	for _, dep := range deps {
		niddle := dep.pkgName + dep.pkgVerOrNetVer
		if _, ok := hashset[niddle]; !ok {
			hashset[niddle] = true
			resultDeps = append(resultDeps, dep)
		}
	}
	return
}

// Sometimes, they use `PkgMicrosoft_VisualStudio_Services_Release_Client_15_131_1` as var name,
// which would yeild Microsoft.VisualStudio.Services.Release.Client.15.131.1 as package name.
// This function could extract the real pkgName and pkgVer from bad pkgName.
func extractPostfixPkgVerFromPkgName(badPkgName string) (realPkgName, pkgVer string) {
	// Reverse returns its argument string reversed rune-wise left to right.
	strings_reverse := func (s string) string {
		r := []rune(s)
		for i, j := 0, len(r)-1; i < len(r)/2; i, j = i+1, j-1 {
			r[i], r[j] = r[j], r[i]
		}
		return string(r)
	}

	splitedReversedStr := strings.Split(strings_reverse(badPkgName), ".")
	for eleIndex, ele := range splitedReversedStr {
		if strings.IndexFunc(ele, func(r rune) bool {return strings.IndexRune("0123456789", r) == -1}) == -1 {
			// There's no non-number characters
			// This element is version number.
		} else {
			pkgVer = strings_reverse(strings.Join(splitedReversedStr[:eleIndex], "."))
			realPkgName = strings_reverse(strings.Join(splitedReversedStr[eleIndex:], "."))
			break
		}
	}

	if realPkgName == "" {
		// This is not a valid PkgName.PkgVer format. Give up the guess.
		// If there's no `.` in pkgVer, someone is using partial version number. Luckily, nuget-download-package can do this now.
		// Example: microsoft.azure.webjobs:2 (PkgMicrosoft_Azure_Webjobs_2)
		// Example: newtonsoft.json.12:net45
		realPkgName = badPkgName
		pkgVer = ""
	}
	return
}

// List file in directory, recursively.
// `find $dirPath -name "$regexToFind"`
// Example: find all .csproj: "^.+\\.(csproj)$"
func FindInDirectory(dirPath, regexToFind string) (resultFilePathes []string, err error) {
	libRegEx, err := regexp.Compile(regexToFind)
	if err != nil {
		return
	}

	err = filepath.Walk(dirPath, func(path string, info os.FileInfo, err error) error {
		if err == nil && libRegEx.MatchString(info.Name()) {
			resultFilePathes = append(resultFilePathes, path)
		}
		return nil
	})

	return
}



