package main

import (
	"flag"
	"fmt"
	"log"
	"os"
)

// Some options here. Would be improved in beta release.
const DEDUCT_PKGNAME_FROM_VARNAME = true
const USE_PROJECT_NETVER_INSTEAD_OF_HINTPATH_NETVER = false
const OPENXT_VERSION = "1.4.1-3"

func print_help_and_exit() {
	println("Usage: openxt <subcommand> [options...]")
	println("Subcommand := sync / sync2 / env")
	println("For 'sync', you must set --local-repo-dir, --project-dir")
	println("For 'sync2', you must set --local-repo-dir, --project-dir")
	println("For 'env', you must set --local-repo-dir, --project-dir, --bin-dir")
	println("")
    println("'sync' would guess package version from your csproj, while 'sync2' would only use package version from packages.config. ")
	println("It doesn't mean that sync2 is better than sync. Both works well in well-written projects. ")
	println("Guideline: If upstream nuget packager sucks, use 'sync2'. If csproj coder sucks, use 'sync'. ")
	flag.PrintDefaults()
	os.Exit(1)
}

func main() {
	log.Println("OpenXT version " + OPENXT_VERSION)

	projectDir := flag.String("project-dir", "", "Path to the CoreXT/OpenXT project. Could be a small subproject, instead of the whole larget CoreXT project. ")
	nugetConfigPath := flag.String("nuget-config", "", "Path to nuget.config. It's ~/.nuget/NuGet/NuGet.Config by default. ")
	localRepoDir := flag.String("local-repo-dir", "", "Path to local nuget repo. (known as CxCache in CoreXT)")
	binDir := flag.String("bin-dir", "", "Path for output build. (TODO: remove in the future)")
	shell := flag.String("shell", "bash", "Shell for the output env variables, supports bash/fish/powershell/cmd")
	if(len(os.Args) < 2) {
		print_help_and_exit()
	}

	// trick
	os.Args = os.Args[1:]
	flag.Parse()

	assertStringNotEmpty(*projectDir, "`--project-dir` must be set")
	assertStringNotEmpty(*localRepoDir, "`--local-repo-dir` must be set")

	switch os.Args[0] {
	case "sync":
		deps := ScanForDependencies(*projectDir)
		SyncPackages(*nugetConfigPath, *localRepoDir, deps)
	case "sync2":
		deps2 := ScanForDependencies2(*projectDir)
		SyncPackages2(*nugetConfigPath, *localRepoDir, deps2)
	case "env":
		assertStringNotEmpty(*binDir, "`--bin-dir` must be set")

		deps := ScanForDependencies(*projectDir)
		envvar := GenerateCorextEnvvar(*localRepoDir, *binDir, deps)
		// log.Println() and println() writes to stderr, and fmt.Println() writes to stdout.
		fmt.Println(SerializeEnvvar(*shell, false, envvar))
	default:
		panic("Expected subcommand `sync`, `sync2` or `env`. ")
	}
}
