package main

import (
	"encoding/json"
	"github.com/antchfx/xmlquery"
	"github.com/mcuadros/go-version"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"sync"
)

/*
`dotnet add` has a bug: 
If my csproj is using net462, and I run
dotnet add package Microsoft.Azure.Kusto.Cloud.Platform --package-directory ~/tmp/ttt --framework net462

It would say: Package Microsoft.Azure.Kusto.Cloud.Platform 9.1.0 is not compatible with net462
However, Microsoft.Azure.Kusto.Cloud.Platform 8.1.5 IS compatible with net462. It doesn't download at all!

dotnet add also requires a valid csproj. It's heavy, complex, and slow. It sucks. 

Let's write a simple one!
*/

func panicErrorIfAny(err error, title string) {
	if err != nil {
		panic("[Error] on " + title + ": " + err.Error())
	}
}
func logErrorIfAny(err error, title string) {
	if err != nil {
		log.Println("[Error] on " + title + ": " + err.Error())
	}
}


func netVersionNugetFormatToStandardFormat(netVersionNugetFormat string) string {
	// nuget format: ".NETFramework4.6.2", ".NETCoreApp2.0", ".NETStandard2.0"
	// standard format: "net462", "netcoreapp2.0", "net5.0", "netstandard2.1"
	// https://docs.microsoft.com/en-us/dotnet/standard/frameworks
	netVersionNugetFormat = strings.ToLower(netVersionNugetFormat)
	if strings.HasPrefix(netVersionNugetFormat, ".netframework") {
		frameworkVer := netVersionNugetFormat[len(".netframework"):]
		return "net" + strings.ReplaceAll(frameworkVer, ".", "")
	} else if strings.HasPrefix(netVersionNugetFormat, ".netcoreapp") {
		return netVersionNugetFormat[1:]
	} else if strings.HasPrefix(netVersionNugetFormat, ".netstandard") {
		return netVersionNugetFormat[1:]
	} else {
		return "netunknown" // do not panic please
	}
}

func httpGet(url, authu, authp, outputFilepath string) string {
	// This function would follow HTTP 30x
	log.Println("GET " + url + " with username " + authu)
	req, err := http.NewRequest("GET", url, nil)
	panicErrorIfAny(err, "http.NewRequest")
	if authu + authp != "" {
		req.SetBasicAuth(authu, authp)
	}
	resp, err := http.DefaultClient.Do(req)
	logErrorIfAny(err, "GET request")
	defer resp.Body.Close()
	respTxt, err := ioutil.ReadAll(resp.Body)
	logErrorIfAny(err, "GET request")
	if outputFilepath != "" {
		err := ioutil.WriteFile(outputFilepath, respTxt, 0777)
		logErrorIfAny(err, "WriteFile " + outputFilepath)
		return ""
	} else {
		return string(respTxt)
	}
}

func initNugetIndexJsonCache(nugetConfigPath, pkgName string) (nugetIndexJsonCache, nugetIndexJsonCache_Login []string) {
	// download index.json from every source, and cache them in array.
	f, err := os.Open(nugetConfigPath)
	panicErrorIfAny(err, "Open " + nugetConfigPath)
	parsed, err := xmlquery.Parse(f)
	panicErrorIfAny(err, "Parse XML " + nugetConfigPath)
	sources := xmlquery.Find(parsed, "//configuration/packageSources/add")

	var wg sync.WaitGroup
	nugetIndexJsonCache_wlock := sync.RWMutex{}
	for _, source := range sources {
		// Parse xml
		sname, surl, sauth_username, sauth_password := "", "", "", ""
		for _, attr := range source.Attr {
			if attr.Name.Local == "key" {
				sname = attr.Value
			} else if attr.Name.Local == "value" {
				surl = attr.Value
			}
		}

		auth_entries := xmlquery.Find(parsed, "//configuration/packageSourceCredentials/" + sname + "/add")
		for _, auth_entry := range auth_entries {
			outputVarForThisEntry := &sauth_username
			for _, attr := range auth_entry.Attr {
				if attr.Name.Local == "key"  && attr.Value == "ClearTextPassword"{
					outputVarForThisEntry = &sauth_password
				} else if attr.Name.Local == "value" {
					*outputVarForThisEntry = attr.Value
				}
			}
		}

		// Send the HTTP request and download index.json!
		wg.Add(1)
		go func(url, authu, authp string) {
			defer wg.Done()

			// This message could be cached, to speedup.
			source_desc_json := httpGet(url, authu, authp, "")
			if len(source_desc_json) == 0 {
				return
			}

			// The golang json parse library sucks. Prevent it from crashing
			//   on a json syntax error.
			defer func() {recover()} ()

			var parsedJson map[string]interface{}
			json.Unmarshal([]byte(source_desc_json), &parsedJson)
			endpoints := parsedJson["resources"].([]interface{})
			selectedEndpoint := ""
			for _, endpoint_ := range endpoints {
				endpoint := endpoint_.(map[string]interface{})
				if endpoint["@type"] == "RegistrationsBaseUrl/Versioned" {
					selectedEndpoint = endpoint["@id"].(string)
				}
			}

			respTxt := httpGet(selectedEndpoint + pkgName + "/index.json", authu, authp, "")
			if len(respTxt) > 0 && ! strings.Contains(respTxt, "Can't find the package ") {
				nugetIndexJsonCache_wlock.Lock()
				nugetIndexJsonCache = append(nugetIndexJsonCache, respTxt)
				nugetIndexJsonCache_Login = append(nugetIndexJsonCache_Login, authu + ":" + authp)
				nugetIndexJsonCache_wlock.Unlock()
			}
		}(surl, sauth_username, sauth_password)
	}
	wg.Wait()
	return
}

// recursive. If package exists, it exit.
// One of [pkgVerBegin, pkgVerEnd] would be downloaded.
func downloadPackageAndDeps(packageName, pkgVerBegin, pkgVerEnd, localRepoDir string, indexJsons []string) {

}

// non-recursive. If package exists, it exit.
// One of [pkgVerBegin, pkgVerEnd] would be downloaded.
func downloadPackage(packageName, pkgVer, localRepoDir string, indexJsons []string) {

}


// [pkgVerBegin, pkgVerEnd] is good for frameworkVersion.
// This function was designed to download dependencies.
func decidePackageVersionRange(packageName, frameworkVersion string, indexJsons []string) (pkgVerBegin, pkgVerEnd string) {
	// Generated by curl-to-Go: https://mholt.github.io/curl-to-go
	// curl -u bensl:xbwejuparq4ighqs4pjq6bqjqmxpzghhtqshql2uy3woi73ew6iq https://o365exchange.pkgs.visualstudio.com/959adb23-f323-4d52-8203-ff34e5cbeefa/_packaging/aacaa93f-61ac-420c-8b20-c17c5d2b52f1/nuget/v3/registrations2-semver2/microsoft.azure.kusto.cloud.platform/index.json


	return "1.1.1", "8.8.8"
}

// It's too hard to parse the json... Let's get everything we need in one time.
// If frameworkVersion != "", then I find the latest available package for this framework.
// If packageVersion != "", then I find the url for this version.
// You must only set ONE OF THEM! If you set both, the `frameworkVersion` would be ignored.
func decidePackageVersion(frameworkVersion, packageVersion string, indexJsons, indexJsons_Login []string, debugOutput bool) (maxAvailableVersion, maxAvailableVersionUrl, maxAvailableVersionUrlLogin string) {
	// The golang json parse library sucks. Prevent it from crashing
	//   on a json syntax error.
	//
	// This function is very ugly. Fucking go-json library and go exception.
	defer func() {recover()} ()

	possibleAvailableVersion, possibleAvailableVersionUrl, PossibleAvailableVersionUrlLogin := "", "", ""

	for indexJson_currIndex, indexJson := range indexJsons {
		func () {
			defer func() { recover() }() // try parse, skip this entry on error
			// entry = xml.findall("/items/[]/items/[]/catalogEntry")
			// entry/version, entry/dependencyGroups[]/targetFramework, entry/../packageContent
			// If there's no `targetFramework` in dependencyGroups, this package depends on specific version of parent package. it sucks.
			var j1 map[string]interface{}
			json.Unmarshal([]byte(indexJson), &j1)
			j2 := j1["items"].([]interface{})
			for _, j2ele := range j2 {
				j3 := j2ele.(map[string]interface{})["items"]
				j4 := j3.([]interface{})
				for _, jPkgVersionItem_ := range j4 {
					func() {
						defer func() { recover() }() // try parse, skip this entry on error
						jPkgVersionItem := jPkgVersionItem_.(map[string]interface{})
						jCatalogEntry := jPkgVersionItem["catalogEntry"].(map[string]interface{})
						myVersionText := jCatalogEntry["version"].(string)
						myVersionUrl := jPkgVersionItem["packageContent"].(string)
						if debugOutput {
							log.Println("DEBUG: reaching " + myVersionText)
						}

						// Let's check if this version is ok
						thisPkgVersionIsOk, thisPkgVersionIsPossibleOk := false, false

						if frameworkVersion != "" {
							if j5, ok := jCatalogEntry["dependencyGroups"]; ok {
								for _, j6 := range j5.([]interface{}) {
									j7 := j6.(map[string]interface{})
									if j8, ok := j7["targetFramework"]; ok {
										if debugOutput {
											log.Println("DEBUG: " + frameworkVersion +" vs "+ netVersionNugetFormatToStandardFormat(j8.(string)))
										}
										if frameworkVersion == netVersionNugetFormatToStandardFormat(j8.(string)) {
											thisPkgVersionIsOk = true
											break
										}
									} else {
										// this pkgVersion has no targetFramework limitation.
										thisPkgVersionIsPossibleOk = true
									}
								}
							} else {
								if debugOutput {
									log.Println("WARNING: This version has no dependency information at all. Can not process, mark as possible OK. ")
								}
								// no dependencyGroups at all
								thisPkgVersionIsPossibleOk = true
							}
						}

						if packageVersion != "" {
							if debugOutput {
								log.Println("DEBUG: " + packageVersion + " vs " + myVersionText)
							}
							// It should be a percise match here. However, someone in ControlPlane using PkgNewtonsoft_json_12.
							// They only provide parts of the version number. It sucks.
							thisPkgVersionIsOk = packageVersion == myVersionText
							// That's why we use a prefix match here.
							thisPkgVersionIsPossibleOk = strings.HasPrefix(myVersionText, packageVersion)

							// Some noob is writing `windowsazure.servicebus:6.0.0.0` (should be 6.0.0).
							// Let's also tolerate this. Fuck!
							thisPkgVersionIsPossibleOk = strings.HasPrefix(packageVersion, myVersionText)
						}

						if thisPkgVersionIsOk {
							// log.Println("DEBUG: good " + myVersionText)
							// This version is ok! Log it.
							if version.CompareSimple(myVersionText, maxAvailableVersion) == 1 {
								// if left > right
								maxAvailableVersion = myVersionText
								maxAvailableVersionUrl = myVersionUrl
								maxAvailableVersionUrlLogin = indexJsons_Login[indexJson_currIndex]
							}
						}
						if thisPkgVersionIsPossibleOk {
							// log.Println("DEBUG: possible good " + myVersionText)
							if version.CompareSimple(myVersionText, possibleAvailableVersion) == 1 {
								// if left > right
								possibleAvailableVersion = myVersionText
								possibleAvailableVersionUrl = myVersionUrl
								PossibleAvailableVersionUrlLogin = indexJsons_Login[indexJson_currIndex]
							}
						}
						if(debugOutput) {
							log.Println("DEBUG: leaving " + myVersionText + ". IsOk=" + strconv.FormatBool(thisPkgVersionIsOk) + ", IsPossibleOk=" + strconv.FormatBool(thisPkgVersionIsPossibleOk))
						}
					}() // catch exception and continue
				}
			}
		}() // catch exception and continue
	}

	if maxAvailableVersion == "" {
		// That's so bad... No confirmed working version, let's use a possible working one.
		maxAvailableVersion = possibleAvailableVersion
		maxAvailableVersionUrl = possibleAvailableVersionUrl
		maxAvailableVersionUrlLogin = PossibleAvailableVersionUrlLogin
	}

	return
}

func main() {
	if(len(os.Args) < 4) {
		log.Println("Usage: nuget-download-package <packageName> <packageVersion/frameworkVersion> <localRepoDir> [nugetConfigPath]")
		log.Println("frameworkVersion maybe 'net472', 'netstandard2.0', 'netcoreapp3.1' ...")
		log.Println("default nugetConfigPath is UserHomeDir/.nuget/NuGet/NuGet.Config")
		os.Exit(1)
	}

    pkgName, pkgVerOrNetVer, localRepoDir, nugetConfigPath := strings.ToLower(os.Args[1]), os.Args[2], os.Args[3], ""
    if len(os.Args) == 4 || os.Args[4] == "" {
    	homeDir, err := os.UserHomeDir()
    	panicErrorIfAny(err, "Get UserHomeDir")
    	nugetConfigPath = filepath.Join(homeDir, ".nuget", "NuGet", "NuGet.Config")
	} else {
		nugetConfigPath = os.Args[4]
	}

    indexJsons, indexJsons_Login := initNugetIndexJsonCache(nugetConfigPath, pkgName)

	pkgVer, targetUrl, targetUrlLogin := "", "", ""
    if strings.HasPrefix(pkgVerOrNetVer, "net") {
    	pkgVer, targetUrl, targetUrlLogin = decidePackageVersion(pkgVerOrNetVer, "", indexJsons, indexJsons_Login, os.Getenv("OPENXT_DEBUG") == "1")
    } else {
		pkgVer, targetUrl, targetUrlLogin = decidePackageVersion("", pkgVerOrNetVer, indexJsons, indexJsons_Login, os.Getenv("OPENXT_DEBUG") == "1")
	}

	if pkgVer == "" {
		log.Println("Error: can not find a valid pkgVer for " + pkgName + ":" + pkgVerOrNetVer)
		os.Exit(2)
	}
	log.Println("Using pkgVer " + pkgVer + " from " + targetUrl)

	pkgBaseDir := filepath.Join(localRepoDir, pkgName, pkgVer)
	if stat, err := os.Stat(filepath.Join(pkgBaseDir, "lib")); err == nil && stat.IsDir() {
		log.Println(pkgBaseDir + "/lib already exists. This version has already been installed. Skipping the installation. ")
		return
	}

	err := os.RemoveAll(pkgBaseDir)
	logErrorIfAny(err, "rm -rf " + pkgBaseDir)
	err = os.MkdirAll(pkgBaseDir, 0777)
	logErrorIfAny(err, "mkdir -p " + pkgBaseDir)

	if targetUrlLogin == "" {
		httpGet(targetUrl, "", "", filepath.Join(pkgBaseDir, "openxt.pkgdown.zip"))
	} else {
		s := strings.Split(targetUrlLogin, ":")
		httpGet(targetUrl, s[0], s[1], filepath.Join(pkgBaseDir, "openxt.pkgdown.zip"))
	}

	err = Extract(filepath.Join(pkgBaseDir, "openxt.pkgdown.zip"), pkgBaseDir)
	logErrorIfAny(err, "unzip openxt.pkgdown.zip")
}

