package main

import (
	"fmt"
	"github.com/antchfx/xmlquery"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"regexp"
	"strings"
)

type dependencyItem struct {
	pkgName, envName, targetNetVer string
}

// This function reads a parsed XML root node, and guess the target dotnet version of this CSPROJ.
// This function is only for internal usage.
func guessNetVerFromXml(parsed *xmlquery.Node) string {
	refs := xmlquery.Find(parsed, "//Project/PropertyGroup/TargetFramework")
	for _, ref := range refs {
		// Looks like net461, netstandard2.0
		return ref.OutputXML(false)
	}
	refs2 := xmlquery.Find(parsed, "//Project/PropertyGroup/TargetFrameworkVersion")
	for _, ref := range refs2 {
		txt := ref.OutputXML(false)
		targetFramework := targetFrameworkVersion2TargetFramework(txt)
		if targetFramework == "" {
			log.Println("Warning: Invalid TargetFrameworkVersion `" + txt + "` detected, rejecting this candidate...")
		} else {
			return targetFramework
		}
	}
	return ""
}

// Parse csproj file, and get list of packages to download
func ParseDependencies(csprojPath string) (dependencies []dependencyItem, err error) {
	f, err := os.Open(csprojPath)
	if err != nil {
		return
	}
	parsed, err := xmlquery.Parse(f)
	if err != nil {
		return
	}
	projectNetVer := guessNetVerFromXml(parsed)
	references := xmlquery.Find(parsed, "//Project/ItemGroup/Reference")
	for _, ref := range references {
		// This codeblock processes one <Reference>
		pkgName, envName, targetNetVer := "", "", ""
		if USE_PROJECT_NETVER_INSTEAD_OF_HINTPATH_NETVER {
			targetNetVer = projectNetVer
		}

		for _, attr := range ref.Attr {
			if attr.Name.Local == "Include" {
				pkgName = attr.Value // GOT
			}
		}
		if pkgName != "" {
			subNode := ref.SelectElement("HintPath")
			if subNode != nil {
				hintPath := subNode.OutputXML(false)
				endPos := strings.Index(hintPath, ")")
				if endPos == -1 {
					log.Println("DEBUG: skip parsing this envName because hintPath is invalid: " + hintPath)
					continue
				}
				envName = hintPath[2:endPos] // GOT

				// Guess version from HintPath may cause problem. We should guess version from project targetFramework.
				if targetNetVer == "" && hintPath[endPos+2:endPos+5] == "lib" {
					hintPath = hintPath[endPos+6:]
					if hintPath[:3] != "net" {
						// Sometimes the hintpath doesn't contain netver. Use project netver. (Example: $(PkgWadi_Heavy_Contracts)\lib\WADI.CoreUtils.dll)
						targetNetVer = projectNetVer
					} else {
						endPos := strings.IndexAny(hintPath, "/\\") // Maybe `/lib/` or `\lib\`
						targetNetVer = hintPath[:endPos]
					}
				}
			}
		}

		// patch1
		if DEDUCT_PKGNAME_FROM_VARNAME {
			// If this compile-time switch is enabled, we guess pkgName from varName, and we're not using
			// pkgName from <Reference Include=xxx>.
			guessedName := guessPkgNameFromVarName(envName) // PkgCis_SdkV2 requires this to work.
			if guessedName != "" {
				pkgName = guessedName
			}
		}

		// patch2
		if targetNetVer == "" {
			targetNetVer = "net472" // Default dotnet version
		}
		dependencies = append(dependencies, dependencyItem{pkgName, envName, targetNetVer})
	}
	return
}

// List all `.csproj` file in directory, recursively.
func ScanCsprojInDirectory(dirPath string) (csprojPathes []string, err error) {
	libRegEx, err := regexp.Compile("^.+\\.(csproj)$")
	if err != nil {
		return
	}

	err = filepath.Walk(dirPath, func(path string, info os.FileInfo, err error) error {
		if err == nil && libRegEx.MatchString(info.Name()) {
			csprojPathes = append(csprojPathes, path)
		}
		return nil
	})

	return
}

// This function panic on error, because it's not recoverable.
// Returns key-value pairs for environment variables to set.
func GenerateCorextEnvvar(localRepoPath, buildOutputPath string, deps []dependencyItem) map[string]string {
	result := make(map[string]string)

	// BEGIN: Prepare temporary fake file for CoreXT imports.
	tmpdir, err := ioutil.TempDir("", "openxt-envvar")
	panicErrorIfAny(err, "GenerateCorextEnvvar create temp dir")

	csprojPluginTemplate := "<Project xmlns=\"http://schemas.microsoft.com/developer/msbuild/2003\"><PropertyGroup><OutputPath>%v</OutputPath></PropertyGroup></Project>"
	csprojPluginFilename := tmpdir + string(os.PathSeparator) + "openxt.plugin.csproj"
	emptyProjectFilename := tmpdir + string(os.PathSeparator) + "Microsoft.CSharp.targets"

	err = ioutil.WriteFile(csprojPluginFilename, []byte(fmt.Sprintf(csprojPluginTemplate, buildOutputPath)), 0777)
	panicErrorIfAny(err, "CreateTempFile")
	err = ioutil.WriteFile(emptyProjectFilename, []byte("<Project></Project>"), 0777)
	panicErrorIfAny(err, "CreateTempFile")

	result["EnvironmentConfig"] = csprojPluginFilename
	result["MSBuildToolsPath"] = tmpdir // MSBuildToolsPath can not be overridden. This statement would have no effect at all.
	result["ExtendedTargetsPath"] = tmpdir
	// END: Prepare temporary fake file for CoreXT imports.

	// Set PkgXxxxx variables.
	for _, dep := range deps {
		if dep.envName != "" {
			pkgPath, err := GetPackagePathFromName(localRepoPath, dep.pkgName, dep.targetNetVer)
			logErrorIfAny(err, "GetPackagePathFromName: ")
			result[dep.envName] = pkgPath
		}
	}

	return result
}
