// Copyright 2014 Citra Emulator Project
// Licensed under GPLv2
// Refer to the license.txt file included.  

#include <map>
#include <string>

#include "common/string_util.h"
#include "common/symbols.h"

#include "core/mem_map.h"

#include "core/hle/kernel/address_arbiter.h"
#include "core/hle/kernel/event.h"
#include "core/hle/kernel/kernel.h"
#include "core/hle/kernel/mutex.h"
#include "core/hle/kernel/shared_memory.h"
#include "core/hle/kernel/thread.h"

#include "core/hle/function_wrappers.h"
#include "core/hle/svc.h"
#include "core/hle/service/service.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// Namespace SVC

namespace SVC {

enum ControlMemoryOperation {
    MEMORY_OPERATION_HEAP       = 0x00000003,
    MEMORY_OPERATION_GSP_HEAP   = 0x00010003,
};

/// Map application or GSP heap memory
Result ControlMemory(u32* out_addr, u32 operation, u32 addr0, u32 addr1, u32 size, u32 permissions) {
    DEBUG_LOG(SVC,"called operation=0x%08X, addr0=0x%08X, addr1=0x%08X, size=%08X, permissions=0x%08X", 
        operation, addr0, addr1, size, permissions);

    switch (operation) {

    // Map normal heap memory
    case MEMORY_OPERATION_HEAP:
        *out_addr = Memory::MapBlock_Heap(size, operation, permissions);
        break;

    // Map GSP heap memory
    case MEMORY_OPERATION_GSP_HEAP:
        *out_addr = Memory::MapBlock_HeapGSP(size, operation, permissions);
        break;

    // Unknown ControlMemory operation
    default:
        ERROR_LOG(SVC, "unknown operation=0x%08X", operation);
    }
    return 0;
}

/// Maps a memory block to specified address
Result MapMemoryBlock(Handle handle, u32 addr, u32 permissions, u32 other_permissions) {
    DEBUG_LOG(SVC, "called memblock=0x%08X, addr=0x%08X, mypermissions=0x%08X, otherpermission=%d", 
        handle, addr, permissions, other_permissions);

    Kernel::MemoryPermission permissions_type = static_cast<Kernel::MemoryPermission>(permissions);
    switch (permissions_type) {
    case Kernel::MemoryPermission::Read:
    case Kernel::MemoryPermission::Write:
    case Kernel::MemoryPermission::ReadWrite:
    case Kernel::MemoryPermission::DontCare:
        Kernel::MapSharedMemory(handle, addr, permissions_type, 
            static_cast<Kernel::MemoryPermission>(other_permissions));
        break;
    default:
        ERROR_LOG(OSHLE, "unknown permissions=0x%08X", permissions);
    }
    return 0;
}

/// Connect to an OS service given the port name, returns the handle to the port to out
Result ConnectToPort(Handle* out, const char* port_name) {
    Service::Interface* service = Service::g_manager->FetchFromPortName(port_name);

    DEBUG_LOG(SVC, "called port_name=%s", port_name);
    _assert_msg_(KERNEL, (service != nullptr), "called, but service is not implemented!");

    *out = service->GetHandle();

    return 0;
}

/// Synchronize to an OS service
Result SendSyncRequest(Handle handle) {
    Kernel::Object* object = Kernel::g_object_pool.GetFast<Kernel::Object>(handle);

    _assert_msg_(KERNEL, (object != nullptr), "called, but kernel object is nullptr!");
    DEBUG_LOG(SVC, "called handle=0x%08X(%s)", handle, object->GetTypeName().c_str());

    bool wait = false;
    Result res = object->SyncRequest(&wait);
    if (wait) {
        Kernel::WaitCurrentThread(WAITTYPE_SYNCH); // TODO(bunnei): Is this correct?
    }

    return res;
}

/// Close a handle
Result CloseHandle(Handle handle) {
    // ImplementMe
    ERROR_LOG(SVC, "(UNIMPLEMENTED) called handle=0x%08X", handle);
    return 0;
}

/// Wait for a handle to synchronize, timeout after the specified nanoseconds
Result WaitSynchronization1(Handle handle, s64 nano_seconds) {
    // TODO(bunnei): Do something with nano_seconds, currently ignoring this
    bool wait = false;
    bool wait_infinite = (nano_seconds == -1); // Used to wait until a thread has terminated

    Kernel::Object* object = Kernel::g_object_pool.GetFast<Kernel::Object>(handle);

    DEBUG_LOG(SVC, "called handle=0x%08X(%s:%s), nanoseconds=%d", handle, object->GetTypeName().c_str(), 
            object->GetName().c_str(), nano_seconds);

    _assert_msg_(KERNEL, (object != nullptr), "called, but kernel object is nullptr!");

    Result res = object->WaitSynchronization(&wait);

    // Check for next thread to schedule
    if (wait) {
        HLE::Reschedule(__func__);
        return 0;
    }

    return res;
}

/// Wait for the given handles to synchronize, timeout after the specified nanoseconds
Result WaitSynchronizationN(s32* out, Handle* handles, s32 handle_count, bool wait_all, 
    s64 nano_seconds) {
    // TODO(bunnei): Do something with nano_seconds, currently ignoring this
    bool unlock_all = true;
    bool wait_infinite = (nano_seconds == -1); // Used to wait until a thread has terminated

    DEBUG_LOG(SVC, "called handle_count=%d, wait_all=%s, nanoseconds=%d", 
        handle_count, (wait_all ? "true" : "false"), nano_seconds);

    // Iterate through each handle, synchronize kernel object
    for (s32 i = 0; i < handle_count; i++) {
        bool wait = false;
        Kernel::Object* object = Kernel::g_object_pool.GetFast<Kernel::Object>(handles[i]);

        _assert_msg_(KERNEL, (object != nullptr), "called handle=0x%08X, but kernel object "
            "is nullptr!", handles[i]);

        DEBUG_LOG(SVC, "\thandle[%d] = 0x%08X(%s:%s)", i, handles[i], object->GetTypeName().c_str(), 
            object->GetName().c_str());

        Result res = object->WaitSynchronization(&wait);

        if (!wait && !wait_all) {
            *out = i;
            return 0;
        } else {
            unlock_all = false;
        }
    }

    if (wait_all && unlock_all) {
        *out = handle_count;
        return 0;
    }

    // Check for next thread to schedule
    HLE::Reschedule(__func__);

    return 0;
}

/// Create an address arbiter (to allocate access to shared resources)
Result CreateAddressArbiter(u32* arbiter) {
    DEBUG_LOG(SVC, "called");
    Handle handle = Kernel::CreateAddressArbiter();
    *arbiter = handle;
    return 0;
}

/// Arbitrate address
Result ArbitrateAddress(Handle arbiter, u32 address, u32 type, u32 value, s64 nanoseconds) {
    return Kernel::ArbitrateAddress(arbiter, static_cast<Kernel::ArbitrationType>(type), address, 
        value);
}

/// Used to output a message on a debug hardware unit - does nothing on a retail unit
void OutputDebugString(const char* string) {
    OS_LOG(SVC, "%s", string);
}

/// Get resource limit
Result GetResourceLimit(Handle* resource_limit, Handle process) {
    // With regards to proceess values:
    // 0xFFFF8001 is a handle alias for the current KProcess, and 0xFFFF8000 is a handle alias for 
    // the current KThread.
    *resource_limit = 0xDEADBEEF;
    ERROR_LOG(SVC, "(UNIMPLEMENTED) called process=0x%08X", process);
    return 0;
}

/// Get resource limit current values
Result GetResourceLimitCurrentValues(s64* values, Handle resource_limit, void* names, 
    s32 name_count) {
    ERROR_LOG(SVC, "(UNIMPLEMENTED) called resource_limit=%08X, names=%s, name_count=%d",
        resource_limit, names, name_count);
    Memory::Write32(Core::g_app_core->GetReg(0), 0); // Normmatt: Set used memory to 0 for now
    return 0;
}

/// Creates a new thread
Result CreateThread(u32 priority, u32 entry_point, u32 arg, u32 stack_top, u32 processor_id) {
    std::string name;
    if (Symbols::HasSymbol(entry_point)) {
        TSymbol symbol = Symbols::GetSymbol(entry_point);
        name = symbol.name;
    } else {
        name = StringFromFormat("unknown-%08x", entry_point);
    }

    Handle thread = Kernel::CreateThread(name.c_str(), entry_point, priority, arg, processor_id,
        stack_top);

    Core::g_app_core->SetReg(1, thread);

    DEBUG_LOG(SVC, "called entrypoint=0x%08X (%s), arg=0x%08X, stacktop=0x%08X, "
        "threadpriority=0x%08X, processorid=0x%08X : created handle=0x%08X", entry_point, 
        name.c_str(), arg, stack_top, priority, processor_id, thread);
    
    return 0;
}

/// Called when a thread exits
u32 ExitThread() {
    Handle thread = Kernel::GetCurrentThreadHandle();

    DEBUG_LOG(SVC, "called, pc=0x%08X", Core::g_app_core->GetPC()); // PC = 0x0010545C

    Kernel::StopThread(thread, __func__);
    HLE::Reschedule(__func__);
    return 0;
}

/// Gets the priority for the specified thread
Result GetThreadPriority(s32* priority, Handle handle) {
    *priority = Kernel::GetThreadPriority(handle);
    return 0;
}

/// Sets the priority for the specified thread
Result SetThreadPriority(Handle handle, s32 priority) {
    return Kernel::SetThreadPriority(handle, priority);
}

/// Create a mutex
Result CreateMutex(Handle* mutex, u32 initial_locked) {
    *mutex = Kernel::CreateMutex((initial_locked != 0));
    DEBUG_LOG(SVC, "called initial_locked=%s : created handle=0x%08X", 
        initial_locked ? "true" : "false", *mutex);
    return 0;
}

/// Release a mutex
Result ReleaseMutex(Handle handle) {
    DEBUG_LOG(SVC, "called handle=0x%08X", handle);
    _assert_msg_(KERNEL, (handle != 0), "called, but handle is nullptr!");
    Kernel::ReleaseMutex(handle);
    return 0;
}

/// Get current thread ID
Result GetThreadId(u32* thread_id, Handle thread) {
    ERROR_LOG(SVC, "(UNIMPLEMENTED) called thread=0x%08X", thread);
    return 0;
}

/// Query memory
Result QueryMemory(void* info, void* out, u32 addr) {
    ERROR_LOG(SVC, "(UNIMPLEMENTED) called addr=0x%08X", addr);
    return 0;
}

/// Create an event
Result CreateEvent(Handle* evt, u32 reset_type) {
    *evt = Kernel::CreateEvent((ResetType)reset_type);
    DEBUG_LOG(SVC, "called reset_type=0x%08X : created handle=0x%08X", 
        reset_type, *evt);
    return 0;
}

/// Duplicates a kernel handle
Result DuplicateHandle(Handle* out, Handle handle) {
    DEBUG_LOG(SVC, "called handle=0x%08X", handle);

    // Translate kernel handles -> real handles
    if (handle == Kernel::CurrentThread) {
        handle = Kernel::GetCurrentThreadHandle();
    }
    _assert_msg_(KERNEL, (handle != Kernel::CurrentProcess),
        "(UNIMPLEMENTED) process handle duplication!");
    
    // TODO(bunnei): FixMe - This is a hack to return the handle that we were asked to duplicate.
    *out = handle;

    return 0;
}

/// Signals an event
Result SignalEvent(Handle evt) {
    Result res = Kernel::SignalEvent(evt);
    DEBUG_LOG(SVC, "called event=0x%08X", evt);
    return res;
}

/// Clears an event
Result ClearEvent(Handle evt) {
    Result res = Kernel::ClearEvent(evt);
    DEBUG_LOG(SVC, "called event=0x%08X", evt);
    return res;
}

/// Sleep the current thread
void SleepThread(s64 nanoseconds) {
    DEBUG_LOG(SVC, "called nanoseconds=%d", nanoseconds);
}

const HLE::FunctionDef SVC_Table[] = {
    {0x00, nullptr,                         "Unknown"},
    {0x01, HLE::Wrap<ControlMemory>,        "ControlMemory"},
    {0x02, HLE::Wrap<QueryMemory>,          "QueryMemory"},
    {0x03, nullptr,                         "ExitProcess"},
    {0x04, nullptr,                         "GetProcessAffinityMask"},
    {0x05, nullptr,                         "SetProcessAffinityMask"},
    {0x06, nullptr,                         "GetProcessIdealProcessor"},
    {0x07, nullptr,                         "SetProcessIdealProcessor"},
    {0x08, HLE::Wrap<CreateThread>,         "CreateThread"},
    {0x09, HLE::Wrap<ExitThread>,           "ExitThread"},
    {0x0A, HLE::Wrap<SleepThread>,          "SleepThread"},
    {0x0B, HLE::Wrap<GetThreadPriority>,    "GetThreadPriority"},
    {0x0C, HLE::Wrap<SetThreadPriority>,    "SetThreadPriority"},
    {0x0D, nullptr,                         "GetThreadAffinityMask"},
    {0x0E, nullptr,                         "SetThreadAffinityMask"},
    {0x0F, nullptr,                         "GetThreadIdealProcessor"},
    {0x10, nullptr,                         "SetThreadIdealProcessor"},
    {0x11, nullptr,                         "GetCurrentProcessorNumber"},
    {0x12, nullptr,                         "Run"},
    {0x13, HLE::Wrap<CreateMutex>,          "CreateMutex"},
    {0x14, HLE::Wrap<ReleaseMutex>,         "ReleaseMutex"},
    {0x15, nullptr,                         "CreateSemaphore"},
    {0x16, nullptr,                         "ReleaseSemaphore"},
    {0x17, HLE::Wrap<CreateEvent>,          "CreateEvent"},
    {0x18, HLE::Wrap<SignalEvent>,          "SignalEvent"},
    {0x19, HLE::Wrap<ClearEvent>,           "ClearEvent"},
    {0x1A, nullptr,                         "CreateTimer"},
    {0x1B, nullptr,                         "SetTimer"},
    {0x1C, nullptr,                         "CancelTimer"},
    {0x1D, nullptr,                         "ClearTimer"},
    {0x1E, nullptr,                         "CreateMemoryBlock"},
    {0x1F, HLE::Wrap<MapMemoryBlock>,       "MapMemoryBlock"},
    {0x20, nullptr,                         "UnmapMemoryBlock"},
    {0x21, HLE::Wrap<CreateAddressArbiter>, "CreateAddressArbiter"},
    {0x22, HLE::Wrap<ArbitrateAddress>,     "ArbitrateAddress"},
    {0x23, HLE::Wrap<CloseHandle>,          "CloseHandle"},
    {0x24, HLE::Wrap<WaitSynchronization1>, "WaitSynchronization1"},
    {0x25, HLE::Wrap<WaitSynchronizationN>, "WaitSynchronizationN"},
    {0x26, nullptr,                         "SignalAndWait"},
    {0x27, HLE::Wrap<DuplicateHandle>,      "DuplicateHandle"},
    {0x28, nullptr,                         "GetSystemTick"},
    {0x29, nullptr,                         "GetHandleInfo"},
    {0x2A, nullptr,                         "GetSystemInfo"},
    {0x2B, nullptr,                         "GetProcessInfo"},
    {0x2C, nullptr,                         "GetThreadInfo"},
    {0x2D, HLE::Wrap<ConnectToPort>,        "ConnectToPort"},
    {0x2E, nullptr,                         "SendSyncRequest1"},
    {0x2F, nullptr,                         "SendSyncRequest2"},
    {0x30, nullptr,                         "SendSyncRequest3"},
    {0x31, nullptr,                         "SendSyncRequest4"},
    {0x32, HLE::Wrap<SendSyncRequest>,      "SendSyncRequest"},
    {0x33, nullptr,                         "OpenProcess"},
    {0x34, nullptr,                         "OpenThread"},
    {0x35, nullptr,                         "GetProcessId"},
    {0x36, nullptr,                         "GetProcessIdOfThread"},
    {0x37, HLE::Wrap<GetThreadId>,          "GetThreadId"},
    {0x38, HLE::Wrap<GetResourceLimit>,     "GetResourceLimit"},
    {0x39, nullptr,                         "GetResourceLimitLimitValues"},
    {0x3A, HLE::Wrap<GetResourceLimitCurrentValues>, "GetResourceLimitCurrentValues"},
    {0x3B, nullptr,                         "GetThreadContext"},
    {0x3C, nullptr,                         "Break"},
    {0x3D, HLE::Wrap<OutputDebugString>,    "OutputDebugString"},
    {0x3E, nullptr,                         "ControlPerformanceCounter"},
    {0x3F, nullptr,                         "Unknown"},
    {0x40, nullptr,                         "Unknown"},
    {0x41, nullptr,                         "Unknown"},
    {0x42, nullptr,                         "Unknown"},
    {0x43, nullptr,                         "Unknown"},
    {0x44, nullptr,                         "Unknown"},
    {0x45, nullptr,                         "Unknown"},
    {0x46, nullptr,                         "Unknown"},
    {0x47, nullptr,                         "CreatePort"},
    {0x48, nullptr,                         "CreateSessionToPort"},
    {0x49, nullptr,                         "CreateSession"},
    {0x4A, nullptr,                         "AcceptSession"},
    {0x4B, nullptr,                         "ReplyAndReceive1"},
    {0x4C, nullptr,                         "ReplyAndReceive2"},
    {0x4D, nullptr,                         "ReplyAndReceive3"},
    {0x4E, nullptr,                         "ReplyAndReceive4"},
    {0x4F, nullptr,                         "ReplyAndReceive"},
    {0x50, nullptr,                         "BindInterrupt"},
    {0x51, nullptr,                         "UnbindInterrupt"},
    {0x52, nullptr,                         "InvalidateProcessDataCache"},
    {0x53, nullptr,                         "StoreProcessDataCache"},
    {0x54, nullptr,                         "FlushProcessDataCache"},
    {0x55, nullptr,                         "StartInterProcessDma"},
    {0x56, nullptr,                         "StopDma"},
    {0x57, nullptr,                         "GetDmaState"},
    {0x58, nullptr,                         "RestartDma"},
    {0x59, nullptr,                         "Unknown"},
    {0x5A, nullptr,                         "Unknown"},
    {0x5B, nullptr,                         "Unknown"},
    {0x5C, nullptr,                         "Unknown"},
    {0x5D, nullptr,                         "Unknown"},
    {0x5E, nullptr,                         "Unknown"},
    {0x5F, nullptr,                         "Unknown"},
    {0x60, nullptr,                         "DebugActiveProcess"},
    {0x61, nullptr,                         "BreakDebugProcess"},
    {0x62, nullptr,                         "TerminateDebugProcess"},
    {0x63, nullptr,                         "GetProcessDebugEvent"},
    {0x64, nullptr,                         "ContinueDebugEvent"},
    {0x65, nullptr,                         "GetProcessList"},
    {0x66, nullptr,                         "GetThreadList"},
    {0x67, nullptr,                         "GetDebugThreadContext"},
    {0x68, nullptr,                         "SetDebugThreadContext"},
    {0x69, nullptr,                         "QueryDebugProcessMemory"},
    {0x6A, nullptr,                         "ReadProcessMemory"},
    {0x6B, nullptr,                         "WriteProcessMemory"},
    {0x6C, nullptr,                         "SetHardwareBreakPoint"},
    {0x6D, nullptr,                         "GetDebugThreadParam"},
    {0x6E, nullptr,                         "Unknown"},
    {0x6F, nullptr,                         "Unknown"},
    {0x70, nullptr,                         "ControlProcessMemory"},
    {0x71, nullptr,                         "MapProcessMemory"},
    {0x72, nullptr,                         "UnmapProcessMemory"},
    {0x73, nullptr,                         "Unknown"},
    {0x74, nullptr,                         "Unknown"},
    {0x75, nullptr,                         "Unknown"},
    {0x76, nullptr,                         "TerminateProcess"},
    {0x77, nullptr,                         "Unknown"},
    {0x78, nullptr,                         "CreateResourceLimit"},
    {0x79, nullptr,                         "Unknown"},
    {0x7A, nullptr,                         "Unknown"},
    {0x7B, nullptr,                         "Unknown"},
    {0x7C, nullptr,                         "KernelSetState"},
    {0x7D, nullptr,                         "QueryProcessMemory"},
};

void Register() {
    HLE::RegisterModule("SVC_Table", ARRAY_SIZE(SVC_Table), SVC_Table);
}

} // namespace
