// Copyright 2018 yuzu Emulator Project
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#include "common/assert.h"
#include "common/common_types.h"
#include "video_core/engines/shader_bytecode.h"
#include "video_core/shader/shader_ir.h"

namespace VideoCommon::Shader {

using Tegra::Shader::Instruction;
using Tegra::Shader::OpCode;
using Tegra::Shader::SubOp;

u32 ShaderIR::DecodeArithmetic(BasicBlock& bb, u32 pc) {
    const Instruction instr = {program_code[pc]};
    const auto opcode = OpCode::Decode(instr);

    Node op_a = GetRegister(instr.gpr8);

    Node op_b = [&]() -> Node {
        if (instr.is_b_imm) {
            return GetImmediate19(instr);
        } else if (instr.is_b_gpr) {
            return GetRegister(instr.gpr20);
        } else {
            return GetConstBuffer(instr.cbuf34.index, instr.cbuf34.offset);
        }
    }();

    switch (opcode->get().GetId()) {
    case OpCode::Id::MOV_C:
    case OpCode::Id::MOV_R: {
        // MOV does not have neither 'abs' nor 'neg' bits.
        SetRegister(bb, instr.gpr0, op_b);
        break;
    }
    case OpCode::Id::FMUL_C:
    case OpCode::Id::FMUL_R:
    case OpCode::Id::FMUL_IMM: {
        // FMUL does not have 'abs' bits and only the second operand has a 'neg' bit.
        UNIMPLEMENTED_IF_MSG(instr.fmul.tab5cb8_2 != 0, "FMUL tab5cb8_2({}) is not implemented",
                             instr.fmul.tab5cb8_2.Value());
        UNIMPLEMENTED_IF_MSG(
            instr.fmul.tab5c68_0 != 1, "FMUL tab5cb8_0({}) is not implemented",
            instr.fmul.tab5c68_0.Value()); // SMO typical sends 1 here which seems to be the default
        UNIMPLEMENTED_IF_MSG(instr.generates_cc,
                             "Condition codes generation in FMUL is not implemented");

        op_b = GetOperandAbsNegFloat(op_b, false, instr.fmul.negate_b);

        // TODO(Rodrigo): Should precise be used when there's a postfactor?
        Node value = Operation(OperationCode::FMul, PRECISE, op_a, op_b);

        if (instr.fmul.postfactor != 0) {
            auto postfactor = static_cast<s32>(instr.fmul.postfactor);

            // Postfactor encoded as 3-bit 1's complement in instruction, interpreted with below
            // logic.
            if (postfactor >= 4) {
                postfactor = 7 - postfactor;
            } else {
                postfactor = 0 - postfactor;
            }

            if (postfactor > 0) {
                value = Operation(OperationCode::FMul, NO_PRECISE, value,
                                  Immediate(static_cast<f32>(1 << postfactor)));
            } else {
                value = Operation(OperationCode::FDiv, NO_PRECISE, value,
                                  Immediate(static_cast<f32>(1 << -postfactor)));
            }
        }

        value = GetSaturatedFloat(value, instr.alu.saturate_d);

        SetRegister(bb, instr.gpr0, value);
        break;
    }
    default:
        UNIMPLEMENTED_MSG("Unhandled arithmetic instruction: {}", opcode->get().GetName());
    }

    return pc;
}

} // namespace VideoCommon::Shader