// Copyright 2014 Citra Emulator Project
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#include "common/common_types.h"

#include "core/mem_map.h"

#include "core/hle/hle.h"
#include "core/hle/kernel/address_arbiter.h"
#include "core/hle/kernel/thread.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// Kernel namespace

namespace Kernel {

ResultVal<SharedPtr<AddressArbiter>> AddressArbiter::Create(std::string name) {
    SharedPtr<AddressArbiter> address_arbiter(new AddressArbiter);
    // TOOD(yuriks): Don't create Handle (see Thread::Create())
    CASCADE_RESULT(auto unused, Kernel::g_handle_table.Create(address_arbiter));

    address_arbiter->name = std::move(name);

    return MakeResult<SharedPtr<AddressArbiter>>(std::move(address_arbiter));
}

ResultCode AddressArbiter::ArbitrateAddress(ArbitrationType type, VAddr address, s32 value,
        u64 nanoseconds) {
    switch (type) {

    // Signal thread(s) waiting for arbitrate address...
    case ArbitrationType::Signal:
        // Negative value means resume all threads
        if (value < 0) {
            ArbitrateAllThreads(address);
        } else {
            // Resume first N threads
            for(int i = 0; i < value; i++)
                ArbitrateHighestPriorityThread(address);
        }
        break;

    // Wait current thread (acquire the arbiter)...
    case ArbitrationType::WaitIfLessThan:
        if ((s32)Memory::Read32(address) <= value) {
            Kernel::WaitCurrentThread_ArbitrateAddress(address);
            HLE::Reschedule(__func__);
        }
        break;
    case ArbitrationType::WaitIfLessThanWithTimeout:
        if ((s32)Memory::Read32(address) <= value) {
            Kernel::WaitCurrentThread_ArbitrateAddress(address);
            GetCurrentThread()->WakeAfterDelay(nanoseconds);
            HLE::Reschedule(__func__);
        }
        break;
    case ArbitrationType::DecrementAndWaitIfLessThan:
    {
        s32 memory_value = Memory::Read32(address) - 1;
        Memory::Write32(address, memory_value);
        if (memory_value <= value) {
            Kernel::WaitCurrentThread_ArbitrateAddress(address);
            HLE::Reschedule(__func__);
        }
        break;
    }
    case ArbitrationType::DecrementAndWaitIfLessThanWithTimeout:
    {
        s32 memory_value = Memory::Read32(address) - 1;
        Memory::Write32(address, memory_value);
        if (memory_value <= value) {
            Kernel::WaitCurrentThread_ArbitrateAddress(address);
            GetCurrentThread()->WakeAfterDelay(nanoseconds);
            HLE::Reschedule(__func__);
        }
        break;
    }

    default:
        LOG_ERROR(Kernel, "unknown type=%d", type);
        return ResultCode(ErrorDescription::InvalidEnumValue, ErrorModule::Kernel, ErrorSummary::WrongArgument, ErrorLevel::Usage);
    }
    return RESULT_SUCCESS;
}

} // namespace Kernel
