// Copyright 2014 Citra Emulator Project
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#include <atomic>

#include <QThread>
#include <QGLWidget>

#include "common/common.h"
#include "common/emu_window.h"
#include "common/thread.h"

class QScreen;
class QKeyEvent;

class GRenderWindow;
class GMainWindow;

class EmuThread : public QThread
{
    Q_OBJECT

public:

    /**
     * Start emulation (on new thread)
     *
     * @warning Only call when not running!
     */
    void run() override;

    /**
     * Allow the CPU to process a single instruction (if cpu is not running)
     *
     * @note This function is thread-safe
     */
    void ExecStep() { exec_cpu_step = true; }

    /**
     * Sets whether the CPU is running 
     *
     * @note This function is thread-safe
     */
    void SetCpuRunning(bool running) { cpu_running = running; }

    /**
     * Allow the CPU to continue processing instructions without interruption
     *
     * @note This function is thread-safe
     */
    bool IsCpuRunning() { return cpu_running; }


    /**
     * Shutdown (permantently stops) the CPU
     */
    void ShutdownCpu() { stop_run = true; };

    /**
     * Waits for the CPU shutdown to complete
     */
    void WaitForCpuShutdown() { shutdown_event.Wait(); }


public slots:
    /**
     * Stop emulation and wait for the thread to finish.
     *
     * @details: This function will wait a second for the thread to finish; if it hasn't finished until then, we'll terminate() it and wait another second, hoping that it will be terminated by then.
     * @note: This function is thread-safe.
     */
    void Stop();

private:
    friend class GMainWindow;

    EmuThread(GRenderWindow* render_window);

    bool exec_cpu_step;
    bool cpu_running;
    std::atomic<bool> stop_run;

    GRenderWindow* render_window;

    Common::Event shutdown_event;

signals:
    /**
     * Emitted when the CPU has halted execution
     *
     * @warning When connecting to this signal from other threads, make sure to specify either Qt::QueuedConnection (invoke slot within the destination object's message thread) or even Qt::BlockingQueuedConnection (additionally block source thread until slot returns)
     */
    void DebugModeEntered();
    
    /**
     * Emitted right before the CPU continues execution
     *
     * @warning When connecting to this signal from other threads, make sure to specify either Qt::QueuedConnection (invoke slot within the destination object's message thread) or even Qt::BlockingQueuedConnection (additionally block source thread until slot returns)
     */
    void DebugModeLeft();
};

class GRenderWindow : public QWidget, public EmuWindow
{
    Q_OBJECT

public:
    GRenderWindow(QWidget* parent, GMainWindow& main_window);

    // EmuWindow implementation
    void SwapBuffers() override;
    void MakeCurrent() override;
    void DoneCurrent() override;
    void PollEvents() override;

    void BackupGeometry();
    void RestoreGeometry();
    void restoreGeometry(const QByteArray& geometry); // overridden
    QByteArray saveGeometry();  // overridden

    void keyPressEvent(QKeyEvent* event) override;
    void keyReleaseEvent(QKeyEvent* event) override;

    void mousePressEvent(QMouseEvent *event) override;
    void mouseMoveEvent(QMouseEvent *event) override;
    void mouseReleaseEvent(QMouseEvent *event) override;

    void ReloadSetKeymaps() override;

    void OnClientAreaResized(unsigned width, unsigned height);

    void OnFramebufferSizeChanged();

public slots:
    void moveContext();  // overridden

private:
    void OnMinimalClientAreaChangeRequest(const std::pair<unsigned,unsigned>& minimal_size) override;

    QGLWidget* child;

    QByteArray geometry;

    GMainWindow& main_window;

    /// Device id of keyboard for use with KeyMap
    int keyboard_id;
};
