// Copyright 2018 yuzu emulator team
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#include <fmt/format.h>
#include "core/file_sys/bis_factory.h"
#include "core/file_sys/registered_cache.h"

namespace FileSys {

BISFactory::BISFactory(VirtualDir nand_root_, VirtualDir load_root_, VirtualDir dump_root_)
    : nand_root(std::move(nand_root_)), load_root(std::move(load_root_)),
      dump_root(std::move(dump_root_)),
      sysnand_cache(std::make_unique<RegisteredCache>(
          GetOrCreateDirectoryRelative(nand_root, "/system/Contents/registered"))),
      usrnand_cache(std::make_unique<RegisteredCache>(
          GetOrCreateDirectoryRelative(nand_root, "/user/Contents/registered"))) {}

BISFactory::~BISFactory() = default;

VirtualDir BISFactory::GetSystemNANDContentDirectory() const {
    return GetOrCreateDirectoryRelative(nand_root, "/system/Contents");
}

VirtualDir BISFactory::GetUserNANDContentDirectory() const {
    return GetOrCreateDirectoryRelative(nand_root, "/user/Contents");
}

RegisteredCache* BISFactory::GetSystemNANDContents() const {
    return sysnand_cache.get();
}

RegisteredCache* BISFactory::GetUserNANDContents() const {
    return usrnand_cache.get();
}

VirtualDir BISFactory::GetModificationLoadRoot(u64 title_id) const {
    // LayeredFS doesn't work on updates and title id-less homebrew
    if (title_id == 0 || (title_id & 0x800) > 0)
        return nullptr;
    return GetOrCreateDirectoryRelative(load_root, fmt::format("/{:016X}", title_id));
}

VirtualDir BISFactory::GetModificationDumpRoot(u64 title_id) const {
    if (title_id == 0)
        return nullptr;
    return GetOrCreateDirectoryRelative(dump_root, fmt::format("/{:016X}", title_id));
}

VirtualDir BISFactory::OpenPartition(BisPartitionId id) const {
    switch (id) {
    case BisPartitionId::CalibrationFile:
        return GetOrCreateDirectoryRelative(nand_root, "/prodinfof");
    case BisPartitionId::SafeMode:
        return GetOrCreateDirectoryRelative(nand_root, "/safe");
    case BisPartitionId::System:
        return GetOrCreateDirectoryRelative(nand_root, "/system");
    case BisPartitionId::User:
        return GetOrCreateDirectoryRelative(nand_root, "/user");
    default:
        return nullptr;
    }
}

VirtualFile BISFactory::OpenPartitionStorage(BisPartitionId id) const {
    Core::Crypto::KeyManager keys;
    Core::Crypto::PartitionDataManager pdm{
        Core::System::GetInstance().GetFilesystem()->OpenDirectory(
            FileUtil::GetUserPath(FileUtil::UserPath::SysDataDir), Mode::Read)};
    keys.PopulateFromPartitionData(pdm);

    switch (id) {
    case BisPartitionId::CalibrationBinary:
        return pdm.GetDecryptedProdInfo();
    case BisPartitionId::BootConfigAndPackage2Part1:
    case BisPartitionId::BootConfigAndPackage2Part2:
    case BisPartitionId::BootConfigAndPackage2Part3:
    case BisPartitionId::BootConfigAndPackage2Part4:
    case BisPartitionId::BootConfigAndPackage2Part5:
    case BisPartitionId::BootConfigAndPackage2Part6: {
        const auto new_id = static_cast<u8>(id) -
                            static_cast<u8>(BisPartitionId::BootConfigAndPackage2Part1) +
                            static_cast<u8>(Core::Crypto::Package2Type::NormalMain);
        return pdm.GetPackage2Raw(static_cast<Core::Crypto::Package2Type>(new_id));
    }
    default:
        return nullptr;
    }
}

} // namespace FileSys
