// Copyright 2015 Citra Emulator Project
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#include <cstring>
#include "common/common_types.h"
#include "common/logging/log.h"
#include "core/hw/hw.h"
#include "core/hw/lcd.h"
#include "core/tracer/recorder.h"

namespace LCD {

Regs g_regs;

template <typename T>
inline void Read(T& var, const u32 raw_addr) {
    u32 addr = raw_addr - HW::VADDR_LCD;
    u32 index = addr / 4;

    // Reads other than u32 are untested, so I'd rather have them abort than silently fail
    if (index >= 0x400 || !std::is_same<T, u32>::value) {
        NGLOG_ERROR(HW_LCD, "Unknown Read{} @ {:#010X}", sizeof(var) * 8, addr);
        return;
    }

    var = g_regs[index];
}

template <typename T>
inline void Write(u32 addr, const T data) {
    addr -= HW::VADDR_LCD;
    u32 index = addr / 4;

    // Writes other than u32 are untested, so I'd rather have them abort than silently fail
    if (index >= 0x400 || !std::is_same<T, u32>::value) {
        NGLOG_ERROR(HW_LCD, "Unknown Write{} {:#010X} @ {:#010X}", sizeof(data) * 8, data, addr);
        return;
    }

    g_regs[index] = static_cast<u32>(data);
}

// Explicitly instantiate template functions because we aren't defining this in the header:

template void Read<u64>(u64& var, const u32 addr);
template void Read<u32>(u32& var, const u32 addr);
template void Read<u16>(u16& var, const u32 addr);
template void Read<u8>(u8& var, const u32 addr);

template void Write<u64>(u32 addr, const u64 data);
template void Write<u32>(u32 addr, const u32 data);
template void Write<u16>(u32 addr, const u16 data);
template void Write<u8>(u32 addr, const u8 data);

/// Initialize hardware
void Init() {
    memset(&g_regs, 0, sizeof(g_regs));
    NGLOG_DEBUG(HW_LCD, "Initialized OK");
}

/// Shutdown hardware
void Shutdown() {
    NGLOG_DEBUG(HW_LCD, "Shutdown OK");
}

} // namespace LCD
