// Copyright 2014 Citra Emulator Project
// Licensed under GPLv2
// Refer to the license.txt file included.

#include "gl_shader_util.h"
#include "common/log.h"

#include <vector>
#include <algorithm>

namespace ShaderUtil {

GLuint LoadShaders(const char* vertex_shader, const char* fragment_shader) {

    // Create the shaders
    GLuint vertex_shader_id = glCreateShader(GL_VERTEX_SHADER);
    GLuint fragment_shader_id = glCreateShader(GL_FRAGMENT_SHADER);

    GLint result = GL_FALSE;
    int info_log_length;

    // Compile Vertex Shader
    DEBUG_LOG(GPU, "Compiling vertex shader.");

    glShaderSource(vertex_shader_id, 1, &vertex_shader, NULL);
    glCompileShader(vertex_shader_id);

    // Check Vertex Shader
    glGetShaderiv(vertex_shader_id, GL_COMPILE_STATUS, &result);
    glGetShaderiv(vertex_shader_id, GL_INFO_LOG_LENGTH, &info_log_length);

    std::vector<char> vertex_shader_error(info_log_length);
    glGetShaderInfoLog(vertex_shader_id, info_log_length, NULL, &vertex_shader_error[0]);

    if (info_log_length > 1) {
        DEBUG_LOG(GPU, "%s", &vertex_shader_error[0]);
    }

    // Compile Fragment Shader
    DEBUG_LOG(GPU, "Compiling fragment shader.");

    glShaderSource(fragment_shader_id, 1, &fragment_shader, NULL);
    glCompileShader(fragment_shader_id);

    // Check Fragment Shader
    glGetShaderiv(fragment_shader_id, GL_COMPILE_STATUS, &result);
    glGetShaderiv(fragment_shader_id, GL_INFO_LOG_LENGTH, &info_log_length);

    std::vector<char> fragment_shader_error(info_log_length);
    glGetShaderInfoLog(fragment_shader_id, info_log_length, NULL, &fragment_shader_error[0]);

    if (info_log_length > 1) {
        DEBUG_LOG(GPU, "%s", &fragment_shader_error[0]);
    }

    // Link the program
    DEBUG_LOG(GPU, "Linking program.");

    GLuint program_id = glCreateProgram();
    glAttachShader(program_id, vertex_shader_id);
    glAttachShader(program_id, fragment_shader_id);
    glLinkProgram(program_id);

    // Check the program
    glGetProgramiv(program_id, GL_LINK_STATUS, &result);
    glGetProgramiv(program_id, GL_INFO_LOG_LENGTH, &info_log_length);

    std::vector<char> program_error(std::max(info_log_length, int(1)));
    glGetProgramInfoLog(program_id, info_log_length, NULL, &program_error[0]);

    if (info_log_length > 1) {
        DEBUG_LOG(GPU, "%s", &program_error[0]);
    }

    glDeleteShader(vertex_shader_id);
    glDeleteShader(fragment_shader_id);

    return program_id;
}

}
