import { parseBlob as metaParseBlob } from 'music-metadata-browser';
import { AudioMimeType, SniffAudioExt, GetArrayBuffer, GetMetaFromFile } from "./utils";
import { DecryptResult } from '@/decrypt/entity';

interface Handler {
    handler: Function;
}

const HandlerMap: { [key: string]: Handler } = {
    "x2m": { handler: ProcessX2M },
    "x3m": { handler: ProcessX3M }
};

export async function Decrypt(file: File, raw_filename: string, raw_ext: string): Promise<DecryptResult> {
    const buffer = new Uint8Array(await GetArrayBuffer(file));
    if (!(raw_ext in HandlerMap)) throw "File type is incorrect!";
    const Handler = HandlerMap[raw_ext];
    let musicDecoded: Uint8Array = Handler.handler(buffer);
    
    const ext = SniffAudioExt(musicDecoded, "m4a");
    const mime = AudioMimeType[ext];
    
    let musicBlob = new Blob([musicDecoded], { type: mime });
    
    const musicMeta = await metaParseBlob(musicBlob);
    
    const info = GetMetaFromFile(raw_filename, musicMeta.common.title, musicMeta.common.artist);

    return {
        picture: "",
        title: info.title,
        artist: info.artist,
        ext: ext,
        album: musicMeta.common.album,
        blob: musicBlob,
        file: URL.createObjectURL(musicBlob),
        mime: mime
    }
}

function ProcessX2M(data: Uint8Array) {
    const x2mHeaderSize = 1024;
    const x2mKey = [120, 109, 108, 121];
    let encryptedHeader = data.slice(0, x2mHeaderSize);
    for (let idx = 0; idx < x2mHeaderSize; idx++) {
        let srcIdx = x2mScrambleTable[idx];
        data[idx] = encryptedHeader[srcIdx] ^ x2mKey[idx % x2mKey.length];
    }
    return data;
}

function ProcessX3M(data: Uint8Array) {
    const x3mHeaderSize = 1024;
    const x3mKey = [51, 57, 56, 57, 100, 49, 49, 49, 97, 97, 100, 53, 54, 49, 51, 57, 52, 48, 102, 52, 102, 99, 52, 52, 98, 54, 51, 57, 98, 50, 57, 50];
    let encryptedHeader = data.slice(0, x3mHeaderSize);
    for (let dstIdx = 0; dstIdx < x3mHeaderSize; dstIdx++) {
        let srcIdx = x3mScrambleTable[dstIdx];
        data[dstIdx] = encryptedHeader[srcIdx] ^ x3mKey[dstIdx % x3mKey.length];
    }
    return data;
}

const x2mScrambleTable = [681, 683, 340, 682, 680, 684, 339, 679, 685, 338, 678, 1023, 0, 341, 686, 337, 677, 1022, 1, 342, 687, 336, 676, 1021, 2, 343, 688, 335, 675, 1020, 3, 344, 689, 334, 674, 1019, 4, 345, 690, 333, 673, 1018, 5, 346, 691, 332, 672, 1017, 6, 347, 692, 331, 671, 1016, 7, 348, 693, 330, 670, 1015, 8, 349, 694, 329, 669, 1014, 9, 350, 695, 328, 668, 1013, 10, 351, 696, 327, 667, 1012, 11, 352, 697, 326, 666, 1011, 12, 353, 698, 325, 665, 1010, 13, 354, 699, 324, 664, 1009, 14, 355, 700, 323, 663, 1008, 15, 356, 701, 322, 662, 1007, 16, 357, 702, 321, 661, 1006, 17, 358, 703, 320, 660, 1005, 18, 359, 704, 319, 659, 1004, 19, 360, 705, 318, 658, 1003, 20, 361, 706, 317, 657, 1002, 21, 362, 707, 316, 656, 1001, 22, 363, 708, 315, 655, 1000, 23, 364, 709, 314, 654, 999, 24, 365, 710, 313, 653, 998, 25, 366, 711, 312, 652, 997, 26, 367, 712, 311, 651, 996, 27, 368, 713, 310, 650, 995, 28, 369, 714, 309, 649, 994, 29, 370, 715, 308, 648, 993, 30, 371, 716, 307, 647, 992, 31, 372, 717, 169, 510, 855, 32, 373, 718, 170, 511, 856, 33, 374, 719, 171, 512, 857, 34, 375, 720, 172, 513, 858, 35, 376, 721, 173, 514, 859, 36, 377, 722, 174, 515, 860, 37, 378, 723, 175, 516, 861, 38, 379, 724, 176, 517, 862, 39, 380, 725, 177, 518, 863, 40, 381, 726, 178, 519, 864, 41, 382, 727, 179, 520, 865, 42, 383, 728, 180, 521, 866, 43, 384, 729, 181, 522, 867, 44, 385, 730, 182, 523, 868, 45, 386, 731, 183, 524, 869, 46, 387, 732, 184, 525, 870, 47, 388, 733, 185, 526, 871, 48, 389, 734, 186, 527, 872, 49, 390, 735, 187, 528, 873, 50, 391, 736, 188, 529, 874, 51, 392, 737, 189, 530, 875, 52, 393, 738, 190, 531, 876, 53, 394, 739, 191, 532, 877, 54, 395, 740, 192, 533, 878, 55, 396, 741, 193, 534, 879, 56, 397, 742, 194, 535, 880, 57, 398, 743, 195, 536, 881, 58, 399, 744, 196, 537, 882, 59, 400, 745, 197, 538, 883, 60, 401, 746, 198, 539, 884, 61, 402, 747, 199, 540, 885, 62, 403, 748, 200, 541, 886, 63, 404, 749, 201, 542, 887, 64, 405, 750, 202, 543, 888, 65, 406, 751, 203, 544, 889, 66, 407, 752, 204, 545, 890, 67, 408, 753, 205, 546, 891, 68, 409, 754, 206, 547, 892, 69, 410, 755, 207, 548, 893, 70, 411, 756, 208, 549, 894, 71, 412, 757, 209, 550, 895, 72, 413, 758, 210, 551, 896, 73, 414, 759, 211, 552, 897, 74, 415, 760, 212, 553, 898, 75, 416, 761, 213, 554, 899, 76, 417, 762, 214, 555, 900, 77, 418, 763, 215, 556, 901, 78, 419, 764, 216, 557, 902, 79, 420, 765, 217, 558, 903, 80, 421, 766, 218, 559, 904, 81, 422, 767, 219, 560, 905, 82, 423, 768, 220, 561, 906, 83, 424, 769, 221, 562, 907, 84, 425, 770, 222, 563, 908, 85, 426, 771, 223, 564, 909, 86, 427, 772, 224, 565, 910, 87, 428, 773, 225, 566, 911, 88, 429, 774, 226, 567, 912, 89, 430, 775, 227, 568, 913, 90, 431, 776, 228, 569, 914, 91, 432, 777, 229, 570, 915, 92, 433, 778, 230, 571, 916, 93, 434, 779, 231, 572, 917, 94, 435, 780, 232, 573, 918, 95, 436, 781, 233, 574, 919, 96, 437, 782, 234, 575, 920, 97, 438, 783, 235, 576, 921, 98, 439, 784, 236, 577, 922, 99, 440, 785, 237, 578, 923, 100, 441, 786, 238, 579, 924, 101, 442, 787, 239, 580, 925, 102, 443, 788, 240, 581, 926, 103, 444, 789, 241, 582, 927, 104, 445, 790, 242, 583, 928, 105, 446, 791, 243, 584, 929, 106, 447, 792, 244, 585, 930, 107, 448, 793, 245, 586, 931, 108, 449, 794, 246, 587, 932, 109, 450, 795, 247, 588, 933, 110, 451, 796, 248, 589, 934, 111, 452, 797, 249, 590, 935, 112, 453, 798, 250, 591, 936, 113, 454, 799, 251, 592, 937, 114, 455, 800, 252, 593, 938, 115, 456, 801, 253, 594, 939, 116, 457, 802, 254, 595, 940, 117, 458, 803, 255, 596, 941, 118, 459, 804, 256, 597, 942, 119, 460, 805, 257, 598, 943, 120, 461, 806, 258, 599, 944, 121, 462, 807, 259, 600, 945, 122, 463, 808, 260, 601, 946, 123, 464, 809, 261, 602, 947, 124, 465, 810, 262, 603, 948, 125, 466, 811, 263, 604, 949, 126, 467, 812, 264, 605, 950, 127, 468, 813, 265, 606, 951, 128, 469, 814, 266, 607, 952, 129, 470, 815, 267, 608, 953, 130, 471, 816, 268, 609, 954, 131, 472, 817, 269, 610, 955, 132, 473, 818, 270, 611, 956, 133, 474, 819, 271, 612, 957, 134, 475, 820, 272, 613, 958, 135, 476, 821, 273, 614, 959, 136, 477, 822, 274, 615, 960, 137, 478, 823, 275, 616, 961, 138, 479, 824, 276, 617, 962, 139, 480, 825, 277, 618, 963, 140, 481, 826, 278, 619, 964, 141, 482, 827, 279, 620, 965, 142, 483, 828, 280, 621, 966, 143, 484, 829, 281, 622, 967, 144, 485, 830, 282, 623, 968, 145, 486, 831, 283, 624, 969, 146, 487, 832, 284, 625, 970, 147, 488, 833, 285, 626, 971, 148, 489, 834, 286, 627, 972, 149, 490, 835, 287, 628, 973, 150, 491, 836, 288, 629, 974, 151, 492, 837, 289, 630, 975, 152, 493, 838, 290, 631, 976, 153, 494, 839, 291, 632, 977, 154, 495, 840, 292, 633, 978, 155, 496, 841, 293, 634, 979, 156, 497, 842, 294, 635, 980, 157, 498, 843, 295, 636, 981, 158, 499, 844, 296, 637, 982, 159, 500, 845, 297, 638, 983, 160, 501, 846, 298, 639, 984, 161, 502, 847, 299, 640, 985, 162, 503, 848, 300, 641, 986, 163, 504, 849, 301, 642, 987, 164, 505, 850, 302, 643, 988, 165, 506, 851, 303, 644, 989, 166, 507, 852, 304, 645, 990, 167, 508, 853, 305, 646, 991, 168, 509, 854, 306];
const x3mScrambleTable = [598, 653, 531, 775, 342, 925, 98, 368, 970, 53, 237, 676, 484, 857, 211, 619, 613, 628, 593, 663, 514, 802, 294, 811, 279, 770, 348, 936, 87, 328, 896, 144, 502, 821, 268, 750, 373, 980, 43, 204, 608, 635, 573, 699, 438, 929, 94, 343, 926, 97, 367, 966, 57, 247, 697, 440, 927, 96, 358, 953, 70, 290, 796, 303, 829, 252, 714, 420, 972, 51, 230, 659, 521, 789, 317, 856, 213, 622, 606, 637, 570, 704, 433, 943, 80, 310, 838, 239, 682, 462, 886, 160, 528, 780, 332, 905, 130, 475, 871, 185, 574, 698, 439, 928, 95, 356, 951, 72, 293, 806, 284, 778, 335, 911, 112, 424, 967, 56, 245, 693, 445, 915, 108, 409, 993, 30, 179, 559, 727, 403, 1002, 21, 157, 522, 788, 318, 858, 210, 618, 615, 626, 595, 660, 520, 793, 311, 844, 231, 661, 517, 797, 302, 828, 254, 717, 416, 981, 42, 200, 600, 646, 554, 732, 398, 1015, 8, 124, 467, 880, 167, 541, 753, 369, 973, 50, 229, 658, 523, 787, 319, 860, 208, 614, 627, 594, 662, 516, 799, 298, 818, 271, 756, 364, 963, 60, 257, 722, 410, 992, 31, 181, 563, 713, 422, 969, 54, 240, 683, 459, 892, 149, 509, 808, 282, 774, 344, 930, 93, 341, 924, 99, 372, 979, 44, 207, 612, 629, 591, 665, 506, 812, 277, 767, 351, 939, 84, 323, 876, 173, 549, 741, 385, 1007, 16, 140, 497, 836, 243, 687, 452, 902, 136, 483, 859, 209, 617, 616, 621, 607, 636, 571, 703, 434, 942, 81, 315, 853, 218, 632, 584, 678, 476, 869, 192, 582, 680, 470, 877, 172, 548, 744, 382, 1003, 20, 156, 519, 794, 307, 833, 248, 700, 437, 931, 92, 338, 917, 106, 396, 1017, 6, 122, 465, 883, 164, 535, 766, 352, 941, 82, 316, 855, 215, 624, 604, 641, 565, 710, 426, 956, 67, 285, 781, 330, 900, 138, 487, 851, 221, 644, 558, 728, 402, 1004, 19, 153, 513, 803, 292, 801, 295, 813, 276, 765, 353, 944, 79, 309, 835, 244, 692, 446, 914, 109, 413, 987, 36, 190, 580, 688, 450, 906, 128, 473, 873, 182, 564, 712, 423, 968, 55, 241, 685, 454, 899, 141, 498, 827, 256, 721, 411, 990, 33, 187, 576, 694, 443, 921, 102, 378, 991, 32, 184, 572, 701, 436, 933, 90, 336, 912, 111, 421, 971, 52, 234, 669, 494, 840, 236, 675, 485, 854, 216, 625, 599, 649, 544, 748, 376, 985, 38, 194, 587, 673, 490, 845, 228, 657, 524, 786, 321, 864, 202, 602, 643, 560, 720, 412, 989, 34, 188, 577, 691, 447, 913, 110, 418, 977, 46, 214, 623, 605, 639, 567, 708, 428, 954, 69, 289, 792, 312, 846, 227, 655, 529, 779, 333, 908, 115, 451, 903, 132, 479, 866, 199, 597, 654, 530, 777, 339, 918, 105, 395, 1018, 5, 121, 464, 884, 162, 533, 769, 349, 937, 86, 327, 890, 152, 512, 804, 287, 790, 314, 850, 223, 648, 547, 745, 381, 1001, 22, 158, 525, 784, 324, 882, 165, 537, 762, 357, 952, 71, 291, 798, 301, 824, 263, 736, 392, 1022, 1, 117, 457, 894, 147, 505, 815, 274, 761, 359, 958, 65, 267, 743, 383, 1005, 18, 151, 511, 805, 286, 785, 322, 870, 186, 575, 696, 441, 923, 100, 374, 982, 41, 197, 592, 664, 508, 809, 281, 772, 346, 934, 89, 334, 910, 113, 429, 950, 73, 296, 814, 275, 764, 354, 946, 77, 305, 831, 250, 706, 431, 947, 76, 304, 830, 251, 711, 425, 957, 66, 278, 768, 350, 938, 85, 326, 888, 155, 518, 795, 306, 832, 249, 702, 435, 940, 83, 320, 861, 206, 610, 633, 583, 679, 471, 875, 174, 550, 739, 389, 1014, 9, 125, 468, 879, 168, 542, 752, 370, 974, 49, 222, 647, 552, 735, 393, 1021, 2, 118, 458, 893, 148, 507, 810, 280, 771, 347, 935, 88, 331, 904, 131, 477, 868, 193, 586, 674, 489, 848, 225, 651, 538, 760, 360, 959, 64, 266, 742, 384, 1006, 17, 145, 503, 820, 269, 751, 371, 975, 48, 220, 640, 566, 709, 427, 955, 68, 288, 791, 313, 847, 226, 652, 536, 763, 355, 948, 75, 300, 823, 264, 738, 390, 1020, 3, 119, 460, 891, 150, 510, 807, 283, 776, 340, 919, 104, 387, 1011, 12, 133, 480, 865, 201, 601, 645, 556, 730, 400, 1010, 13, 134, 481, 863, 203, 603, 642, 562, 716, 417, 978, 45, 212, 620, 611, 631, 585, 677, 478, 867, 198, 596, 656, 526, 783, 325, 887, 159, 527, 782, 329, 898, 142, 499, 826, 258, 723, 408, 994, 29, 178, 557, 729, 401, 1009, 14, 135, 482, 862, 205, 609, 634, 579, 689, 449, 907, 127, 472, 874, 180, 561, 719, 414, 986, 37, 191, 581, 684, 455, 897, 143, 501, 822, 265, 740, 388, 1012, 11, 129, 474, 872, 183, 569, 705, 432, 945, 78, 308, 834, 246, 695, 442, 922, 101, 377, 988, 35, 189, 578, 690, 448, 909, 114, 444, 920, 103, 386, 1008, 15, 139, 488, 849, 224, 650, 540, 755, 365, 964, 59, 255, 718, 415, 983, 40, 196, 590, 667, 496, 837, 242, 686, 453, 901, 137, 486, 852, 219, 638, 568, 707, 430, 949, 74, 299, 819, 270, 754, 366, 965, 58, 253, 715, 419, 976, 47, 217, 630, 588, 671, 492, 842, 233, 668, 495, 839, 238, 681, 463, 885, 161, 532, 773, 345, 932, 91, 337, 916, 107, 406, 998, 25, 171, 546, 746, 380, 997, 26, 175, 551, 737, 391, 1023, 0, 116, 456, 895, 146, 504, 817, 272, 758, 362, 961, 62, 260, 725, 405, 999, 24, 170, 545, 747, 379, 996, 27, 176, 553, 733, 397, 1016, 7, 123, 466, 881, 166, 539, 757, 363, 962, 61, 259, 724, 407, 995, 28, 177, 555, 731, 399, 1013, 10, 126, 469, 878, 169, 543, 749, 375, 984, 39, 195, 589, 670, 493, 841, 235, 672, 491, 843, 232, 666, 500, 825, 262, 734, 394, 1019, 4, 120, 461, 889, 154, 515, 800, 297, 816, 273, 759, 361, 960, 63, 261, 726, 404, 1000, 23, 163, 534];
